// **************************************************************************
// * File:   Mem.cpp											   			*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Code to implement memory management using free-lists.			*
// * Note:   This is to improve the speed of generated code since the		*
// *         Microsoft memory management code is quite slow.				*
// **************************************************************************

//---------
// Includes
//---------

#include "Ertsys.hpp"
#include <cstring>
#include <cassert>
#include <new>

//#define _dMemLimit	(1)
//#define _dMemNew		(1)			// set this if we want to break on failed memory allocations

#if _dThreadSafe
  #include <windows.h>	// for 'CRITICAL_SECTION'
#endif

//-----
// Code
//-----

// Note that we have used a namespace directive here instead of a class, so that we can logically group our freelist functionality together.
namespace _eMem
{
  #if _dThreadSafe
	CRITICAL_SECTION memCritSect;
  #endif

	//-------------------------------------------
	// Parameters to control the use of freelists
	//-------------------------------------------

	// SizeIncrement: our quantum of memory allocation
	// Must be a power of 2 and big enough to hold a "void*".
#if defined(_MSC_VER)
	// In MSVC++ / Windows 2000 the operating system will always give multiples of
	// 16 bytes, thus the optimum size for our quantum is also 16 bytes.
	// Note that as our runtime library is built with MSVC++ this will be the quantum
	// for all windows builds - this is probably what we want as this is presumably an operating system issue.
	const unsigned int SizeIncrement = 16;
#else
	// In GCC 3.3 / Linux the operating system quantum is 8 bytes.
	const unsigned int SizeIncrement = 8;
#endif

	// MaxFreeListSize: Size of chunk in the largest freelist we keep.
	// Must be a multiple of SizeIncrement.
	const unsigned int MaxFreeListSize = 1024;

	// live: Flag to prevent use of freelists before CTOR called or after DTOR called. Gets around initialisation order problems.
	static bool live = false;

	// The total amount of memory we have allocated from the external C++ run-time system,
	// as used for freelist-managed memory (i.e. NOT including that allocated directly using '::new').
	static _eSize managedTotalAllocated = 0;

	// The total amount of memory we have allocated from the external C++ run-time system,
	// as used for non-freelist-managed memory (i.e. only that allocated directly using '::new').
	static _eSize unmanagedTotalAllocated = 0;

	// The total amount of memory in our freelists.
	static _eSize totalInFreelists = 0;

	// Number of times we have returned memory to the system (for statistical use only)
	static unsigned int numReleases = 0;

	// Clear specified memory region
	static inline void clear(void* p, _eSize s)
	{
		memset(p, 0, s);
	}

	// Make the stats accessable to the outside world - required by the environment function 'getMemoryUsed()'
	_eSize getManagedTotalAllocated()
	{
		return managedTotalAllocated;
	}

	_eSize getUnmanagedTotalAllocated()
	{
		return unmanagedTotalAllocated;
	}

	_eSize getTotalInFreelists()
	{
		return totalInFreelists;
	}

#if defined(_dMemLimit)
	inline void checkMemLimit()
	{
		if (managedTotalAllocated + unmanagedTotalAllocated > 800*1024*1024)
		{
 #if defined(_MSC_VER)
			__asm int 3
 #elif defined(__GNUC__)
			__asm__
			( "int $3" );
 #endif
		}
	}
#endif

	//----------------------------------
	// A class for maintaining freelists
	//----------------------------------

	class FreeLists
	{
		//-------------------------------------------------
		// A class to represent a single freelist
		// All functions are declared inline for efficiency
		//-------------------------------------------------

		class oneFreeList
		{
		 public:
			void *list;
			unsigned int numFree;

			// Constructor
			oneFreeList()
			{
				list = NULL;
				numFree = 0;
			}

			// Allocate from freelist, returning NULL if nothing there
			void* allocate()
			{
				void* ptr = list;
				if (ptr != NULL)
				{
					list = *reinterpret_cast<void**>(ptr);
					--numFree;
				}
				return ptr;
			}

			// Release memory to freelist
			void release(void *ptr)
			{
				*reinterpret_cast<void**>(ptr) = list;
				list = ptr;
				++numFree;
			}

			// Return all the memory in the freelists to the external C++ memory manager
			void returnAllToSystem()
			{
				void *ptr;
				while ((ptr = allocate()) != NULL)
				{
					:: operator delete(ptr);
				}
				assert(numFree == 0);
			}

			// Destructor
			~oneFreeList()
			{
			  #if !defined(NDEBUG)
				// Free all the memory so that the Microsoft debug runtime system doesn't report memory leaks
				returnAllToSystem();
			  #endif
			}
		};

		// the actual freelists
		oneFreeList lists[MaxFreeListSize/SizeIncrement];

	 public:
		// Constructor
		FreeLists()
		{
			// member "lists" will get initialised just before we get
			// here, so we can signal we are ready to use it...
			managedTotalAllocated = 0;
			unmanagedTotalAllocated = 0;
			totalInFreelists = 0;
			live = true;
		}

		// Destructor
		~FreeLists()
		{
			// member "lists" will get destroyed just after we get here, so signal that the freelists are not to be used
			live = false;
		}

		void returnAllToSystem()
		{
			for (unsigned int i = 0; i < MaxFreeListSize/SizeIncrement; ++i)
			{
				lists[i].returnAllToSystem();
			}
			managedTotalAllocated -= totalInFreelists;	// this much less is outstanding
			totalInFreelists = 0;  						// nothing left in freelists
			++numReleases;
		}

		// Allocate memory (note that this does not clear it)
		void* allocate(_eSize sz)
		{
			if (sz <= MaxFreeListSize && sz != 0)
			{
				const _eInt index = (sz - 1)/SizeIncrement;
				const _eSize roundedUpSize = (index + 1) * SizeIncrement;
				if (live)
				{
					void* ptr = lists[index].allocate();
					if (ptr != NULL)
					{
						totalInFreelists -= roundedUpSize;
						return ptr;
					}
				}

				// To avoid excessive memory use, if we have lots of unused memory in the freelists,
				// release it all (or some of it) to the system.
				// We could choose many different ways of deciding how much unused memory is acceptable
				// and how much to release, depending on whether a significant pause is acceptable.
				// For now, we decide that we allow at most one quarter of the memory to be wasted,
				// and of we exceed this, we free all of it.
				const _eSize maxAllowedWastage = managedTotalAllocated/4;
				if (totalInFreelists > maxAllowedWastage)
				{
					returnAllToSystem();
				}

				// We need to allocate from the global heap, but round up the size so that we can recycle the memory back to our freelist later
				managedTotalAllocated += roundedUpSize;  // assume we will be successful

#if defined(_dMemLimit)
				checkMemLimit();
#endif
				void* p = ::operator new(roundedUpSize, std::nothrow);
				if (p != 0)
				{
					return p;
				}

				returnAllToSystem();

#if defined(_dMemNew)
				void* p = ::operator new(roundedUpSize, std::nothrow);
				if (p != 0)
				{
					return p;
				}

				__asm int 3;
				throw std::bad_alloc();
#else
				return ::operator new(roundedUpSize);		// throws std::bad_alloc if it fails
#endif
			}
			else
			{
				// Here if size is zero or too big for our freelists
				unmanagedTotalAllocated += sz;
#if defined(_dMemLimit)
				checkMemLimit();
#endif
				try
				{
					return ::operator new(sz);
				}
				catch (const std::bad_alloc&) {}

				returnAllToSystem();

#if defined(_dMemNew)
				try
				{
					return ::operator new(sz);
				}
				catch (const std::bad_alloc&)
				{
					__asm int 3;
					throw;
				}
#else
				return ::operator new(sz);
#endif
			}
		}

		// Free memory
		void free(void* ptr, _eSize sz)
		{
			if (sz <= MaxFreeListSize && sz != 0 && live)
			{
				const _eInt index = (sz - 1)/SizeIncrement;
				const _eSize roundedUpSize = (index + 1) * SizeIncrement;
				lists[index].release(ptr);
				totalInFreelists += roundedUpSize;
			}
			else
			{
				:: operator delete(ptr);
				unmanagedTotalAllocated -= sz;
			}
		}
	};

	// This is our free-list manager object. We call allocate and free
	// on this, and it deals with whether or not to allocate from the
	// heap or from recycled memory...
	static FreeLists myFreelists;

	//--------------------------------------------------------------
	// Define public functions we can call using our _eMem namespace
	//--------------------------------------------------------------

	// Allocate memory
	void* alloc(_eSize sz)
	{
	  #if _dThreadSafe
		static bool initdCritSect = false;
		if(!initdCritSect)
		{
			initCritSectLocker();
			initdCritSect = true;
		}
		EnterCriticalSection(&memCritSect);
	  #endif
		void* p = _eMem::myFreelists.allocate(sz);
		_eMem::clear(p, sz);
	  #if _dThreadSafe
		LeaveCriticalSection(&memCritSect);
	  #endif
		return p;
	}

	// Release memory
	void free(void* ptr, _eSize sz)
	{
	  #if _dThreadSafe
		EnterCriticalSection(&memCritSect);
	  #endif
	  #if _dExtraRuntimeChecks
		// '_dExtraRuntimeChecks' is an extra check against using objects after
		// they have been deallocated. We clear the memory, making it more likely
		// that accessing the object will raise a C++ runtime exception...
		_eMem::clear(ptr, sz);
	  #endif
		// Invoke our de-allocator to free the memory (hopefully back to a
		// free-list) ...
		_eMem::myFreelists.free(ptr, sz);
	  #if _dThreadSafe
		LeaveCriticalSection(&memCritSect);
	  #endif
	}

	// Garbage collect (return memory in free lists to system)
	void garbageCollect()
	{
		myFreelists.returnAllToSystem();
	}

  #if _dThreadSafe
	//
	bool initCritSectLocker()
	{
//		try
//		{
			InitializeCriticalSection(&_eMem::memCritSect);
//		}
//		catch(const int critSectExcep)
//		{
//			if(critSectExcep == STATUS_NO_MEMORY)
//				return false;
//			return false;	// anyway...
//		}
		return true;
	}
  #endif

}			                                                    // end namespace _eMem

// End.
