// **************************************************************************
// * File:   Environment.cpp												*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Code to implement interface to the local OS environment (ie.	*
// *         the local filesystem). All primitives used in the code are		*
// *         POSIX complient, and where necessary we provide macros to		*
// *         vendor specific calls.											*
// **************************************************************************

//---------
// Includes
//---------

#include "Ertsys.hpp"

// Regular, standard includes

#include <cerrno>			// for ENOENT etc.
#include <cstdlib>			// for _MAX_PATH etc.
#include <cwchar>
#include <ctime>
#include <sys/types.h>
#include <sys/stat.h>

#include "CharHelpers.hpp"

// Includes and macros to provide POSIX compatability (the macros defined in the Microsoft section allow us to
// use the POSIX names for POSIX standard identifiers in the code)

#if defined(_MSC_VER)

  #include "NtSystemCalls.hpp"
  #include <io.h>			// for the MS specific version of the POSIX 'access' function, '_access'
  #include <sys/utime.h>	// for some reason, MS have moved this standard include to the 'sys' sub-directory
  #include <direct.h>		// for '_getcwd'
  #define utimbuf _utimbuf
  #define S_ISDIR(mode) (mode & _S_IFDIR)
  #define S_IWUSR _S_IWRITE
  #define S_IRUSR _S_IREAD
  #define NAME_MAX _MAX_PATH
  #define LINE_MAX 2048
  // Macro to correct the converted local time to UTC
  #define timezoneadjust(t)	(t - _timezone)

#elif defined(__BCPLUSPLUS__)

  #include <io.h>			// for the MS specific version of the POSIX 'access' function, '_access'
  #include <utime.h>
  #include <direct.h>		// for '_getcwd'
  #include "NtSystemCalls.hpp"

  #define utime _utime
  #define NAME_MAX _MAX_PATH
  #define LINE_MAX 2048
  // Macro to correct the converted local time to UTC
  #define timezoneadjust(t)	(t - _timezone)

#elif defined(__GNUC__)

  #include <unistd.h>		// the POSIX include for the 'access' function
  #include <utime.h>
  #include <dirent.h>		// the POSIX include for the 'NAME_MAX' constant
  #include "LinuxSystemCalls.hpp"

  // Macro to correct the converted local time to UTC
  #define timezoneadjust(t)	(t - __timezone)
  #define _stat stat
#endif

#include "EnvironmentSupport_1.hpp"
#include "LowLevelSupport_1.hpp"

#include "EnvironmentSupport_2.hpp"
#include "LowLevelSupport_2.hpp"

//-----
// Code
//-----

//---------------------------------------------
// Constructor (pass argv[0] as the parameter)
//---------------------------------------------
Environment :: Environment(const _eNativeChar * exepath) : stdinHandle(), stdoutHandle(), stderrHandle()
{
    _mConstructor;
	const _eHndlUnionVoid<FilePath> _vTemp(_eosInterface :: normalizefilename(_mCstr(""), exepath).Ptr());
	ImagePath = _vTemp.Ptr() == NULL ? _mString("<error>") : _mIs(_vTemp.Ptr(), FilePath)->pathName;
	_eosInterface :: initialize();
}

//------------------------------------------------------------------------------------------------
// Environment functions to return file references for standard in input, output and error streams
//------------------------------------------------------------------------------------------------
_eHndl<_eStandardInputStream> Environment :: stdIn() const
{
	if (stdinHandle.IsNull())
	{
		const _eHndl<FileRef> fref(new FileRef(stdin, false));
		stdinHandle = new _eStandardInputStream(fref.Ptr(), this);
	}
	return stdinHandle;
}

_eHndl<_eStandardOutputStream> Environment :: stdOut() const
{
	if (stdoutHandle.IsNull())
	{
		const _eHndl<FileRef> fref(new FileRef(stdout, true));
		stdoutHandle = new _eStandardOutputStream(fref.Ptr(), this);
	}
	return stdoutHandle;
}

_eHndl<_eStandardOutputStream> Environment :: stdErr() const
{
	if (stderrHandle.IsNull())
	{
		const _eHndl<FileRef> fref(new FileRef(stderr, true));
		stderrHandle = new _eStandardOutputStream(fref.Ptr(), this);
	}
	return stderrHandle;
}

//-------------------------------------------------------------------------------
// Function: Environment :: open(_rstring filename, _eSet<FileModeType> mode, _eUnion& ret)
// Purpose:  Open the specified file in the specified mode. Result is a union
//           of types 'FileRef' and 'FileError', indicating the open file
//           or an error code (see Environment.hpp).
//-------------------------------------------------------------------------------
void Environment :: open(const _rstring filename, const _eSet < FileModeType :: _eEnum > mode, _eUnion& ret)
{
	const _eCstring fn(filename);
	const bool appendToExistingFile = mode._ovIn (FileModeType :: append) && fileValid(filename);
	const _eNativeChar* md =	mode._ovIn (FileModeType :: create)
								? mode._ovIn (FileModeType :: text)
									? (mode._ovIn (FileModeType :: read) ? _mCstr("wt+") : _mCstr("wt"))		// create & text
    								: (mode._ovIn (FileModeType :: read) ? _mCstr("wb+") : _mCstr("wb"))		// create & not text
    							: mode._ovIn (FileModeType :: append)
								? appendToExistingFile
									? (mode._ovIn (FileModeType :: text) ? _mCstr("rt+") : _mCstr("rb+"))		// append (file already exists - must open for read access anyway)
									: mode._ovIn (FileModeType :: text)											// append (file does not already exist so open in write mode)
    									? (mode._ovIn (FileModeType :: read) ? _mCstr("wt+") : _mCstr("wt"))	// append & text
    									: (mode._ovIn (FileModeType :: read) ? _mCstr("wb+") : _mCstr("wb"))	// append & not text
								: mode._ovIn (FileModeType :: text)
								? _mCstr("rt")
								: _mCstr("rb");																	// read only
	FILE* f = _mFopen(fn.str(), md);
	if (f == NULL)
	{
		FileError::_eEnum openError;
		openError = ( errno == ENOENT ? FileError::fileNotFound :
					  errno == EACCES || errno == EROFS ? FileError::attribError :
					  FileError::otherError
					);	// decide which error
		ret = _eUnion(_mWrapStorableEnum(openError, FileError));
	}
	else
	{
		FileRef* fref = new FileRef(f, mode._ovIn (FileModeType :: create) || mode._ovIn (FileModeType :: append));
		// If we were opening in append mode and the file already exists, then we must seek to the end of the file
		// because it will have been opened in "r+" mode in order that it is fully writable.
		if (appendToExistingFile)
		{
			fseek(fref->getFileHandle(), 0L, SEEK_END);		//??? we should check for a seek error here!
		}
		ret = _eUnion(fref);
	}
}

// Note: bodies of the methods for opening sockets are in the file 'sockets.cpp'. This is to try to
// reduce executable size for project not requiring sockets. Nigel, 21-3-02.

//-------------------------------------------------------------------------------------
// Function: Environment :: makeDirectory(const _rstring pathname, FileError :: _eEnum & err)
// Purpose:  Create directory specified by 'pathname', including all components that
//           may not already exist. Works for UNC paths too.
// Returns:  If the call fails on a particular path segment, then all segments upto
//           the failed one will have been created, and an error code appropriate to
//           the failed segment will be returned in 'err' ...
//-------------------------------------------------------------------------------------
void Environment :: makeDirectory(const _rstring pathname, FileError :: _eEnum & err)
{
    _mSchema (makeDirectory);
    _mDontShare (this);
	if (pathname._oHash() == 0)
	{
		// asked to create nothing, so count this as a success
		err = FileError :: success;
		return;
	}
	const _eNativeChar pathSepChar = _eosInterface :: getPathSeparator();
	const _eHndlUnionVoid < FilePath > normalised (normalizeFile (((pathSepChar == pathname.last ())
			? pathname
			: pathname.append (pathSepChar)), _mString ("")).Ptr ());
    if (normalised.Ptr () == NULL)
	{
		// couldn't normalize the path, so return with an error...
		err = FileError :: otherError;
		return;
	}
	// The rest of this function is written in 'Perfect'
	_eEnvironmentSupport::makeDirectory(_mIsNotNull (normalised.Ptr (), FilePath) -> pathName, pathSepChar, err);
	// Final instance assignment: self'! = self;
}

//-------------------------------------------------------------------------------------------
// Function: Environment :: open( const _eSeq < _eChar > hostname, const _eInt port,
//                                const SocketMode::_eEnum sktMode, _eUnion & s )
// Purpose:  Open a socket on the host with the specified name ...
//-------------------------------------------------------------------------------------------

// This schema has been moved to file 'Sockets.cpp' for the purposes of efficient linking -
// if the schema is not referenced in user code, then module 'Sockets' will not get linked
// in ...

//-------------------------------------------------------------------------------------------
// Function: Environment :: open( const _eSeq < _eByte > ipAddress, const _eInt port,
//                                const SocketMode::_eEnum sktMode, _eUnion & s )
// Purpose:  Open a socket on the host with the spwcified IP address ...
//-------------------------------------------------------------------------------------------

// This schema has been moved to file 'Sockets.cpp' for the purposes of efficient linking -
// if the schema is not referenced in user code, then module 'Sockets' will not get linked
// in ...

//-----------------------------------------------------------------------------------------------------------------
// Function: Environment :: readLine (_eSeq < _eChar > & line, FileError :: _eEnum & ret)
// Purpose:  Read a line of text from STDIN (usually the keyboard)
//-----------------------------------------------------------------------------------------------------------------
void Environment :: readLine (_eSeq < _eChar > & line, FileError :: _eEnum & ret)
{
    _mSchema (readLine);
    _mDontShare (this);
    _eNativeChar lineBuf[LINE_MAX];
	_eNativeChar *getRes =
	_mFgets(lineBuf, LINE_MAX, stdin);
	if (getRes == NULL)
	{
		line = _mString("");
		if (errno == EBADF)
		{
			ret = FileError :: attribError;		// couldn't read from stream.
			return;
		}
		else
		{
			ret = FileError :: otherError;		// other miscellaneous error.
			return;
		}
	}
	// Deal with removing any line end sequence, ie. CR,LF or LF
	size_t len = _mStrlen(getRes);
	if (getRes[len - 1] == 0x0a && len >= 2 && getRes[len - 2] == 0x0d)
	{
		getRes[len - 2] = 0;		// null terminate at CR in CR,LF sequence
	}
	else if (getRes[len - 1] == 0x0a)
	{
		getRes[len - 1] = 0;		// null terminate at LF
	}
	ret = FileError :: success;
    line = _lString(getRes);
}

//-----------------------------------------------------------------------------------------------------------------
// Function: Environment :: readLine (const FileRef * f, _eSeq < _eChar > & line, FileError :: _eEnum & ret)
// Purpose:  Read a line of text from the file specified (ie. a data block ending in either LF or EOF). Sets the
//           error return on failure ...
//-----------------------------------------------------------------------------------------------------------------
void Environment :: readLine (const FileRef* f, _eSeq<_eChar>& line, FileError :: _eEnum& ret)
{
    _mSchema (readLine);
    _eNativeChar lineBuf[LINE_MAX];
	FILE * const file = f->getFileHandle();
	_eNativeChar *getRes = _mFgets(lineBuf, LINE_MAX, file);
	if (getRes == NULL)
	{
		line = _mString("");
		if (feof(file))
		{
			ret = FileError :: endOfFile;		// reached end of file
		}
		else if (errno == EBADF)
		{
			ret = FileError :: attribError;		// file wasn't opened for reading.
		}
		else
		{
			ret = FileError :: otherError;		// other miscellaneous error.
		}
	}
	else
	{
		line = _lString(lineBuf);
		// keep reading more characters until the line ends with a line feed, or we can't read any more.
		// note: we only need to check s is non-empty due to the possibility of nulls occurring in the file -
		//       this case should be fixed and then this test can be removed.
		for(; 0 < line._oHash() && +(line.last()) != 10 ;)
		{
			getRes = _mFgets(lineBuf, LINE_MAX, file);
			if(getRes == NULL)
			{
				if(feof(file))
				{
					break;
				}
				else
				{
					ret = FileError :: readError;
					return;
				}
			}
			else
			{
				line._osPlusPlus(_lString(lineBuf));
			}
		}
		ret = FileError :: success;
	}
}

//--------------------------------------------------------------------------------------------------------------------------
// Function: Environment :: move(const _eSeq<_eChar> oldPath, const _eSeq<_eChar> newPath, const _eBool clobber, FileError :: _eEnum &res)
// Purpose:  Move the file specified by oldPath to newPath, overwriting if the flag says to allow this. Directories
//           cannot be moved by this call. If the call fails, the error return 'res' is set to one of the following
//           values: 'fileNotFound' if 'oldPath' does not specify an existing file; 'attribError' if 'newPath'
//           specifies an existing file and the 'clobber' argument was false; 'deleteError' if we failed to delete the
//           existing target when trying to clobber it; 'fileSpecError' if 'oldPath' and 'newPath' specify the same file
//           or if either are invalid; or 'otherError' for any other problem (for example, if 'oldPath' specifies a
//           directory) ...
// Notes:    Only the 'last-accessed' attributes on the moved file is changed; all others are preserved.
//--------------------------------------------------------------------------------------------------------------------------
void Environment :: move(const _eSeq<_eChar> oldPath, const _eSeq<_eChar> newPath, const _eBool clobber, FileError :: _eEnum &res)
{
    _mSchema (move);
	_rstring oldPathNorm, newPathNorm;

	// Normalize and error check oldPath and newPath ...
	{
		const _eHndlUnionVoid<FilePath> oldTemp(normalizeFile(_mString(""), oldPath).Ptr());
		const _eHndlUnionVoid<FilePath> newTemp(normalizeFile(_mString(""), newPath).Ptr());
		if (oldTemp.Ptr() == NULL || newTemp.Ptr() == NULL || oldTemp == newTemp)
		{
			// Failed to normalize one or both of the filepaths, or they specified the same file!
			res = FileError :: fileSpecError;
			return;
		}
		oldPathNorm = _mIs(oldTemp.Ptr(), FilePath)->pathName._oPlusPlus(_mIs(oldTemp.Ptr(), FilePath)->fileName);
		newPathNorm = _mIs(newTemp.Ptr(), FilePath)->pathName._oPlusPlus(_mIs(newTemp.Ptr(), FilePath)->fileName);
	}
	{
		_eUnion sfs(fileStatus(oldPathNorm));
		if (_mWithin(sfs.Ptr(), FileStats) && _mIs(sfs.Ptr(), FileStats)->attribs._ovIn (FileAttribute::directory))
		{
			// Source 'file' is actually a directory!
			res = FileError :: otherError;
			return;
		}
	}
	// Check if the target file exists; error if it does and the clobber parameter wasn't set, or delete it ...
	if (fileValid(newPathNorm))
	{
		if (!clobber)
		{
			// Target exists and we weren't instructed to clobber it!
			res = FileError :: attribError;
			return;
		}
		FileError :: _eEnum delRes;
		_rdelete(newPathNorm, delRes);
		if (delRes != FileError :: success)
		{
			// Target exists and although we were instructed to clobber it, we couldn't!
			res = FileError :: deleteError;
			return;
		}
	}
	// OK, the way is clear to attempt the move ...
	if (_mRename(_eCstring(oldPathNorm).str(), _eCstring(newPathNorm).str()))
	{
		// The actual move failed. We check for specific errors by looking at 'errno' which
		// will be set as follows:
		// EACCES if file or directory specified by newname could not be created (invalid path); or
		//        oldname is a directory and newname specifies a different path;
		// ENOENT if file or path specified by oldname not found;
		// EINVAL if one of the names contains invalid characters.
		//??? Actually, errno doesn't always seem to get set correctly - in the case of the newpath (directory)
		//    not existing, the errno seems to get set to ENOENT instead of EACCES, and as a result we return
		//    error 'fileNotFound' instead of 'otherError' in that circumstance.
		res = (errno == ENOENT ? FileError::fileNotFound : errno == EINVAL ? FileError::fileSpecError : FileError::otherError);
		return;
	}
	// Wow! we got here, so we must have succeeded!!!
	res = FileError :: success;
}

//------------------------------------------------------------------------
// Function: Environment :: flush(const FileRef* f, FileError &ret)
// Purpose:  Flush the specified file buffer. Result is an error code
//           of type 'FileError' (see Environment.hpp).
//------------------------------------------------------------------------
void Environment :: flush(const FileRef* f, FileError :: _eEnum &ret)
{
	_mSchema(flush);
	ret = fflush(f->getFileHandle()) == 0
			? FileError :: success
			: FileError :: flushError;
}

//--------------------------------------------------------------------------
// Function: Environment :: close(const FileRef* f, FileError &ret)
// Purpose:  Close the specified file. Result is of type 'FileError',
//           indicating the result code (see Environment.hpp).
//--------------------------------------------------------------------------
void Environment :: close(const FileRef* f, FileError :: _eEnum &ret)
{
	_mSchema(close);
	ret = fclose(f->getFileHandle()) == 0
			? FileError :: success
			: FileError :: closeError;
}

//-------------------------------------------------------------------------------
// Function: Environment :: _rdelete(_rstring filename, FileError& ret)
// Purpose:  Delete the file specified by the full pathname. Result is of type
//           'FileError', indicating the result of the operation. Note that
//           this will fail if the file is currently open at the point of call;
//           if the file is read only; or if the pathname points to either a
//           directory or a non-existant file.
// Notes:    The reason for failure could be determined by checking 'errno' for
//           the values 'EACCES' (file was read only) or 'ENOENT' (the pathname
//           points to a directory or non-existant file). We could return a
//           different failure code depending on the failure indicated by errno.
// Extra:	 Note also that the file really ought to be closed when delete is called
//			 since the FileRef for such an open file would become invalid if the
//			 extrernal-file had been deleted. Ada has a related problem (see the
//			 file environment.adb for details). Note that our current specification
//			 of the file system in builtin.e does not provide a way to determine
//			 whether or not a file is open.
//-------------------------------------------------------------------------------
void Environment :: _rdelete(const _rstring pathname, FileError :: _eEnum &ret)
{
	const _eCstring pn(pathname);
	if (_mUnlink(pn.str()) == -1)
		ret = (errno == ENOENT ? FileError :: fileNotFound : FileError :: deleteError);    		// failure - see note above.
	else
		ret = FileError :: success;
}

//----------------------------------------------------------------------------------------
// Function: Environment :: seek(const FileRef* f, _eNat fpos, FileError &ret)
// Purpose:  Set the file pointer in the specified file. Result
//           is of type 'FileError', indicating the result code (see Environment.hpp).
//----------------------------------------------------------------------------------------
void Environment :: seek(const FileRef* f, _eNat posn, FileError :: _eEnum &ret)
{
	_mSchema(seek);
	ret = (
#if defined(_WIN64)
			_fseeki64(f->getFileHandle(), posn, SEEK_SET)
#else
			fseek(f->getFileHandle(), posn, SEEK_SET)
#endif
			!= 0)
			? FileError :: seekError
			: FileError :: success;
}

//----------------------------------------------------------------------------------------
// Function: Environment :: fastForward(const FileRef * f, FileError &ret)
// Purpose:  Set the file pointer in the specified file to the end of the file.
//           Result is of type 'FileError', indicating the result code (see
//           Environment.hpp).
//----------------------------------------------------------------------------------------
void Environment :: fastForward(const FileRef * f, FileError :: _eEnum &ret)
{
	_mSchema(fastForward);
	ret = (fseek(f->getFileHandle(), 0L, SEEK_END) != 0)		// Seek to end of file
			? FileError :: seekError
			: FileError :: success;
}

//----------------------------------------------------------------------------------------
// Function: Environment :: rewind(const FileRef * f, FileError &ret)
// Purpose:  Set the file pointer in the specified file to the start of the file.
//           Result is of type 'FileError', indicating the result code (see
//           Environment.hpp).
//----------------------------------------------------------------------------------------
void Environment :: rewind(const FileRef * f, FileError :: _eEnum &ret)
{
	_mSchema(rewind);
	seek(f, 0, ret);
}

//--------------------------------------------------------------------------
// Function: Environment :: fileValid(const _rstring pathname) const
// Purpose:  Check if the file specified by the full path given is accessible on the local file
//			 system (ie. whether an open in read mode or a fileStatus operation would succeed) ...
//--------------------------------------------------------------------------
_eBool Environment :: fileValid(const _rstring pathname) const
{
	const _eCstring pn(pathname);
	return _mAccess(pn.str(), 0) == 0;
}

//------------------------------------------------------------------------------------------------------
// Function: Environment :: fileStatus(const _rstring pathname) const
// Purpose:  Return status information about a file, including the times it was created, last modified
//			 and last accessed; its size; its attributes, ie. read, write, directory (size will be 0
//			 if this attribute is returned).
//			 Returns the FileStats object containing the information described, or a FileError.
// Notes:	 The times returned are "Time's", ie. they are in coordinated-universal-time format.
//			 Note that if the path passed ends in a path-separator character, this is taken as an
//			 assertion that the path refers to a directory, and not a file, and so if the path actually
//			 refers to a file instead of a directory, the error 'directoryNotFound' is returned ...
//------------------------------------------------------------------------------------------------------
_eUnion Environment :: fileStatus(const _rstring pathname) const
{
	struct _stat buffer;				// stat's result holder
	// Deal with allocating a string buffer 2 characters longer than required, to hold the converted
	// Perfect string, since we may need to add 1 or 2 characters to it
	// We use an _eCstring to convert the Perfect string to a C string, and then allocate a new buffer
	// with space for 2 extra characters, copying the converted string to it. This is in case we need
	// to append "." or ".\" to it (under MSW only, to fix the cases where the C 'stat' function fails,
	// ie. the "C:" and "C:\" cases, which need to be changed to "C:." and "C:\." respectively). We
	// allocate this space regardless of OS, it's only a couple of bytes afterall
	_eCstring pn(pathname);
	_eNativeChar* cpn = new _eNativeChar[pn.strlen() + 3];	// allocate space for 2 extra chars (and a string terminator)
	_mStrcpy(cpn, pn.str());
	_eCstring npn(cpn);		// build a memory managed string holder (simplifies things when we want to return, mid function)
	delete[] cpn;			// delete the dynamically allocated buffer, since the the _eCstring object now has a copy of its contents
	cpn = npn.changeStr();	// get a pointer to the managed buffer (which contains space for 2 extra characters, remember!)
	// Detect and deal with the case described above, where under MSW, certain directories fail to stat correctly
  #if defined(_MSC_VER)
	if (_mStrlen(cpn) == 2 && cpn[1] == ':')								// eg. "C:"
	{	// Need to append a '.' to the path to ensure the stat function succeeds
		cpn[2] = '.';
		cpn[3] = '\0';
	}
	else if (_mStrlen(cpn) == 3 && cpn[1] == ':' && cpn[2] == '\\')		// eg. "C:\"
	{	// Need to append ".\" to the path to ensure the stat function succeeds
		cpn[3] = '.';
		cpn[4] = '\\';
		cpn[5] = '\0';
	}
  #endif
	// Detect the case where the user has asserted that a path should refer to a directory, and NOT a file
	bool bMustBeDirectory = false;
	{
		size_t lastCharIndex = _mStrlen(cpn) - 1;
		if (cpn[lastCharIndex] == _eosInterface :: getPathSeparator())
		{	// Dump the trailing path-seperator (because this would cause the 'stat(..)' call to fail) and note
			// the fact that the user has asserted the path is a directory
			cpn[lastCharIndex] = '\0';
			bMustBeDirectory = true;
		}
	}
	// OK, do the stat, dealing with the case where the caller asserted the path should be a directory
	if (_mStat(cpn, &buffer) == 0)
	{
		// Obtain info about creation/mod/accessed times

		struct tm *timeTemp;

		timeTemp = gmtime(&buffer.st_ctime);		// Converts the time value (time_t) to a struct tm*
		if (timeTemp == NULL)
		{
			return _eUnion(_mWrapStorableEnum(FileError :: otherError, FileError));	// creation time before Midnight on 1st Jan 1970
		}
		const _eHndl < Time > created(new Time(timeTemp->tm_sec, timeTemp->tm_min, timeTemp->tm_hour, timeTemp->tm_wday, timeTemp->tm_mday, timeTemp->tm_mon + 1, timeTemp->tm_year + 1900));

		timeTemp = gmtime(&buffer.st_mtime);		// Converts the time value (time_t) to a struct tm*
		if (timeTemp == NULL)
		{
			return _eUnion(_mWrapStorableEnum(FileError :: otherError, FileError));	// creation time before Midnight on 1st Jan 1970
		}
		const _eHndl < Time > modified(new Time(timeTemp->tm_sec, timeTemp->tm_min, timeTemp->tm_hour, timeTemp->tm_wday, timeTemp->tm_mday, timeTemp->tm_mon + 1, timeTemp->tm_year + 1900));

		timeTemp = gmtime(&buffer.st_atime);		// Converts the time value (time_t) to a struct tm*
		if (timeTemp == NULL)
		{
			return _eUnion(_mWrapStorableEnum(FileError :: otherError, FileError));	// creation time before Midnight on 1st Jan 1970
		}
		const _eHndl < Time > accessed(new Time(timeTemp->tm_sec, timeTemp->tm_min, timeTemp->tm_hour, timeTemp->tm_wday, timeTemp->tm_mday, timeTemp->tm_mon + 1, timeTemp->tm_year + 1900));

		// Obtain info about file attributes (??? can't check for faArchive, faSystem, faHidden)
		{
			_eSet<FileAttribute :: _eEnum> attribs = _eSet<FileAttribute :: _eEnum>();
			bool bDirectory = S_ISDIR(buffer.st_mode) > 0;
			if (bMustBeDirectory && !bDirectory)
			{	// Caller asserted that the path should refer to a directory (by having the path end in a path-sep char),
				// but it actually refers to a file, so we generate this special error
				return _eUnion(_mWrapStorableEnum(FileError :: directoryNotFound, FileError));
			}
			else
			{
				if (bDirectory)
				{
					attribs = attribs.append(FileAttribute :: directory);
				}
				if (buffer.st_mode & S_IWUSR)
				{
					attribs = attribs.append(FileAttribute :: write);
				}
				if (buffer.st_mode & S_IRUSR)
				{
					attribs = attribs.append(FileAttribute :: read);
				}

				// Generate the file-stat object ...
				return _eUnion
						(	new FileStats
								(	created.Ptr(),						// Creation time
									modified.Ptr(),						// Last modification time
									accessed.Ptr(),						// Last access time
									attribs,							// Attributes
									bDirectory ? 0 : buffer.st_size		// File size (or 0 if the path refers to a directory)
								)
						);
			}
		}
	}
	// Path didn't refer to any existing file or directory ...
	return _eUnion(_mWrapStorableEnum((errno == ENOENT ? FileError :: fileNotFound : FileError :: otherError), FileError));
}


//------------------------------------------------------------------------------------------------------------------------
// Set the attributes of a file
//	schema !setMode(fname: string, atts: set of FileAttribute, res!: out FileError)
//		pre atts <<= set of FileAttribute{read@FileAttribute, write@FileAttribute, execute@FileAttribute}
//------------------------------------------------------------------------------------------------------------------------
void Environment :: setMode(const _rstring filename, const _eSet<FileAttribute :: _eEnum> atts, FileError :: _eEnum &ret)
{
	_mSchema(setMode);

	const _eCstring pn(filename);
	ret = _eosInterface :: chmod(pn.str(), atts);
}

//------------------------------------------------------------------------------------------------------------------------
// Function: Environment :: forceFileTime ( const _rstring filename,
//											const _eAny * modified, const _eAny * accessed,
//											FileError & res)
// Purpose:  Force one of the time attibutes (modified, accessed) on the specified file on the local filesystem.
//			 The schema will fail if the file doesn't exist or is currently open
// Note:	 The two time fields are in fact the 'Perfect' type 'Time || void', where a null value indicates that time
//			 attribute should not be changed.
//------------------------------------------------------------------------------------------------------------------------
void Environment :: forceFileTime ( const _eSeq < _eChar > filename,
									const Time * modified, const Time * accessed,
									FileError :: _eEnum & res
								  )
{
	_mSchema(forceFileTime);

	struct _stat buffer;				// stat's result holder
	const _eCstring pn(filename);
	const _eNativeChar *cpn = pn.str();

	struct utimbuf newTimes;

	// Default the fields in the newTimes temporary variable to the current times on the file so that we
	// can preserve fields that we are not setting
	if (_mStat(cpn, &buffer) == 0)
	{
		newTimes.modtime = buffer.st_mtime;
		newTimes.actime = buffer.st_atime;
	}
	else
	{
		res = FileError :: fileNotFound;
		return;
	}

	// Deal with the time the file was last modified
	if (modified != NULL)
	{
        const _eHndl < Time > nnModified (_mIsNotNull (modified, Time));
        _mBeginEmbeddedAssert _mCheckEmbeddedAssert ((!(nnModified -> year < 1900)), "296,9");

		struct tm tempTime;

		tempTime.tm_sec = static_cast<int>(nnModified->seconds);
		tempTime.tm_min = static_cast<int>(nnModified->minutes);
		tempTime.tm_hour = static_cast<int>(nnModified->hours);
		tempTime.tm_wday = static_cast<int>(nnModified->day);			//??? don't think we need to set this field
		tempTime.tm_mday = static_cast<int>(nnModified->date);
		tempTime.tm_mon = static_cast<int>(nnModified->month - 1);
		tempTime.tm_year = static_cast<int>(nnModified->year - 1900);

		// This next field is required by the standard, and if it is not set, results are undefined.
		// MSDN extract:
		// It must be positive if daylight saving time is in effect; 0 if daylight saving time is not in effect; or
		// negative if status of daylight saving time is unknown. Note that the C run-time library assumes the
		// United Statess rules for implementing the calculation of Daylight Saving Time (DST)
		tempTime.tm_isdst = 0;

		newTimes.modtime = timezoneadjust(mktime(&tempTime));
	}

	// Deal with the time the file was last accessed
	if (accessed != NULL)
	{
        const _eHndl < Time > nnAccessed (_mIsNotNull (accessed, Time));
		_mBeginEmbeddedAssert _mCheckEmbeddedAssert(!(nnAccessed->year < 1900), "321,9");

		struct tm tempTime;

		tempTime.tm_sec = static_cast<int>(nnAccessed->seconds);
		tempTime.tm_min = static_cast<int>(nnAccessed->minutes);
		tempTime.tm_hour = static_cast<int>(nnAccessed->hours);
		tempTime.tm_wday = static_cast<int>(nnAccessed->day);			//??? don't think we need to set this field
		tempTime.tm_mday = static_cast<int>(nnAccessed->date);
		tempTime.tm_mon = static_cast<int>(nnAccessed->month - 1);
		tempTime.tm_year = static_cast<int>(nnAccessed->year - 1900);

		// This next field is required by the standard, and if it is not set, results are undefined.
		// MSDN extract:
		// It must be positive if daylight saving time is in effect; 0 if daylight saving time is not in effect; or
		// negative if status of daylight saving time is unknown. Note that the C run-time library assumes the
		// United Statess rules for implementing the calculation of Daylight Saving Time (DST)
		tempTime.tm_isdst = 0;

		newTimes.actime = timezoneadjust(mktime(&tempTime));
	}

	if (_mUtime(cpn, &newTimes) != 0)
	{
		switch (errno)
		{
			case ENOENT:
			{
				res = FileError :: fileNotFound;
				return;
			}
			default:
			{
				res = FileError :: otherError;
				return;
			}
		}
	}

	res = FileError :: success;
}

//----------------------------------------------------------------------------------------
// Function: Environment :: fileSize(const FileRef* f, _eNat &fsize, FileError &ret)
// Purpose:  Obtain the number of readable characters in the specified file. Result
//           is of type 'FileError', indicating the result code (see
//           Environment.hpp). Note that if the file was not opened in binary mode,
//           then CR, LF sequences may be ignored by the target OS. Therefore, it is
//           safest to always open files in binary mode if this function is to be called.
//----------------------------------------------------------------------------------------
void Environment :: fileSize(const FileRef* f, _eNat &size, FileError :: _eEnum &ret) const
{
	_mSchema(fileSize);

	FILE * const file = f->getFileHandle();
   	const long oldPtr = ftell(file); 				// Get the current file pointer
	if (oldPtr == -1L)
	{
		size = 0;
		ret = FileError :: seekError;
	}
	else if (fseek(file, 0L, SEEK_END) != 0)		// Seek to end of file
	{
		size = 0;
		ret = FileError :: seekError;
	}
	else
	{
   		const long lsize = ftell(file);				// Find the new file pointer
		if (lsize == -1L)
		{
			size = 0;
			ret = FileError :: seekError;
		}
		else
		{
			size = lsize;               	    	// return the size
			ret = (fseek(file, oldPtr, SEEK_SET) != 0)
				? FileError :: seekError			// restore the original file pointer and return
				: FileError :: success;
		}
	}
}

//----------------------------------------------------------------------------------------
// Function: _eUnion Environment :: tell(const FileRef* f)
// Purpose:  Get the current file pointer. Result
//           is of type 'FileError', indicating the result code (see
//           Environment.hpp). Note that if the file was not opened in binary mode,
//           then CR, LF sequences may be ignored by the target OS. Therefore, it is
//           safest to always open files in binary mode if this function is to be called.
//----------------------------------------------------------------------------------------
_eUnion Environment :: tell(const FileRef* f) const
{
	_mSchema(tell);

#if defined(_Win64)
   	const __int64 lpos = _ftelli64(f->getFileHandle());
#else
   	const long lpos = ftell(f->getFileHandle());
#endif
	if (lpos == -1)
	{
		return _eUnion (_mWrapStorableEnum (FileError :: seekError, FileError));
	}
	else
	{
		return _eUnion (_mWrapBasic (static_cast<_eInt>(lpos), _eInt));
	}
}

// Macro for all file writing methods - if the last operation was a read, must flush the file before writing again
// Note: although the documentation claims a flush is sufficient, in fact it seems (at least under MSVC) that a seek is needed
static inline void fileWriteFlush(const FileRef* f)
{
	if (!f->lastOpWasWrite)
	{   fseek(f->getFileHandle(), 0L, SEEK_CUR);
		f->lastOpWasWrite = true;
	}
}

// Similar macro for all file reading methods
static inline void fileReadFlush(const FileRef* f)
{
	if (f->lastOpWasWrite)
	{	fseek(f->getFileHandle(), 0L, SEEK_CUR);
		f->lastOpWasWrite = false;
	}
}

//------------------------------------------------------
// Function: Environment :: print(<type>)
// Purpose:  Write the specified character type to STDOUT.
//------------------------------------------------------
void Environment :: print(const _rstring s)
{
	_mSchema(print);
	const _eCstring temp(s);
#if defined(_dUnicode)
	printf("%ls", temp.str());
#else
	printf("%s", temp.str());
#endif
	fflush(stdout);
}

void Environment :: print(const _eChar c)
{
	_mSchema(print);
	putchar(static_cast<int>((_eUnsignedChar)c));
}

//---------------------------------------------
// Function: Environment :: printStdErr(string)
// Purpose:  Write string to STDERR.
//---------------------------------------------
void Environment :: printStdErr(const _rstring s)
{
	_mSchema(printStdErr);
	const _eCstring temp(s);
#if defined(_dUnicode)
	fprintf(stderr, "%ls", temp.str());
#else
	fprintf(stderr, "%s", temp.str());
#endif
	fflush(stderr);
}

//---------------------------------------------------------------------------
// Function: Environment :: print(const FileRef* f, <type> &, FileError &ret)
// Purpose:  Write the specified character type to file.
//---------------------------------------------------------------------------
void Environment :: print(const FileRef* f, const _rstring s, FileError :: _eEnum &ret)
{
	_mSchema(print);
	fileWriteFlush(f);
	const _eCstring temp(s);
#if defined(_dUnicode)
	char* buf;
	wcs2ncs(buf, temp.str());
	int nw = fprintf(f->getFileHandle(), "%s", buf);
	size_t expected = strlen(buf);
#else
	int nw = fprintf(f->getFileHandle(), "%s", temp.str());
	size_t expected = strlen(temp.str();
#endif
	ret = (nw >= 0 && static_cast<size_t>(nw) == expected)
			? FileError :: success
			: FileError :: writeError;
}

void Environment :: print(const FileRef* f, const _eChar c, FileError :: _eEnum &ret)
{
	_mSchema(print);
	fileWriteFlush(f);
	const int temp(static_cast<int>((_eUnsignedChar)c));
	ret = putc(temp, f->getFileHandle()) == temp
			? FileError :: success
			: FileError :: writeError;
}

//---------------------------------------------------------------------------
// Function: Environment :: write(const FileRef* f, <type> &, FileError &ret)
// Purpose:  Write the specified type to file in binary format.
//---------------------------------------------------------------------------
void Environment :: write(const FileRef* f, const _eSeq<_eByte> s, FileError :: _eEnum &ret)
{
	_mSchema(write);
	fileWriteFlush(f);
	const size_t localBufferSize = 128;
	_eByte localBuffer[localBufferSize];
	const size_t len = static_cast<size_t>(s._oHash());
	const bool allocate = (len > localBufferSize);
	_eByte *temp = allocate
					? static_cast<_eByte*>(_eMem::alloc(len * sizeof(_eByte)))
					: localBuffer;

	for (size_t n = 0; n < len; n++)
	{
		temp[n] = s[n];
	}
	ret = (fwrite(temp, sizeof(_eByte), len, f->getFileHandle()) == len)
			? FileError :: success
			: FileError :: writeError;
	if (allocate)
	{
		_eMem::free(temp, len * sizeof(_eByte));
	}
}

// Given an integer, split it up into the specified number of bytes (i.e. chars) in most-significant byte first order.
// Precondition: upb(buffer) >= numBytes
void Environment :: genEightBitSeq(const _eInt n, const _eInt numBytes, _eByte* const buffer) const
{
	// Shifting a signed integer right in C++ does not guarantee to do a signed shift, therefore if the number of bytes to
	// write is greater than the number of bytes in an _eInt, we need to generate extra 0 or -1 bytes as appropriate.
	int excessBytes = static_cast<int>(numBytes) - sizeof(_eInt);
	if (excessBytes > 0)
	{
		// Generate any excess 0 or -1 bytes that are needed
		for (int i = 0; i < excessBytes; ++i)
		{
			buffer[i] = (n < 0) ? '\xFF' : '\x00';
		}
	}
	else
	{
		excessBytes = 0;
	}

	_eInt temp = n;
	for (int i = static_cast<int>(numBytes); i > excessBytes; )
	{
		--i;
		buffer[i] = static_cast<_eByte>(temp & 0xFF);
		temp = temp >> 8;									// onto next (more significant) byte
	}
}

// This writes integers of specified number-of-bytes in big-endian (MS Byte first) format.
void Environment :: write(const FileRef* f, const _eInt n, const _eInt numBytes, FileError :: _eEnum &ret)
{
	_mSchema(write);
	_mBeginPre _mCheckPre(numBytes > 0, "0,0");

	fileWriteFlush(f);
	FILE * const file = f->getFileHandle();

	if (numBytes == 1)
	{
		// Optimisation for the case of writing a single byte
		int temp = static_cast<int>(n) & 0xff;									// remove any higher bits so the error check is valid
		ret = (putc(temp, file) == temp)
			? FileError :: success
			: FileError :: writeError;
	}
	else
	{
		const int localBufferSize = 2 * sizeof(_eInt);
		_eByte localBuffer[localBufferSize];										// we use this buffer if numBytes is small enough
		const bool allocate = (static_cast<int>(numBytes) > localBufferSize);
		_eByte* const buffer = (allocate)
								? static_cast<_eByte *>(::_eMem::alloc(static_cast<_eSize>(numBytes)))
								: localBuffer;

		genEightBitSeq(n, numBytes, buffer);									// convert the integer into the specified number of bytes.

		ret = (fwrite(buffer, static_cast<size_t>(numBytes), 1, file) == 1)		// write the sequence of bytes to the file
				? FileError :: success
				: FileError :: writeError;

		if (allocate)
		{
			_eMem::free(buffer, static_cast<_eSize>(numBytes));					// release the buffer
		}
	}
}

void Environment :: write(const FileRef* f, const _eInt n, FileError :: _eEnum &ret)
{
	_mSchema(write);
	fileWriteFlush(f);
	ret = (fwrite(&n, sizeof(_eInt), 1, f->getFileHandle()) == 1)
			? FileError :: success
			: FileError :: writeError;
}

void Environment :: write(const FileRef* f, const _eReal r, FileError :: _eEnum &ret)
{
	_mSchema(write);
	fileWriteFlush(f);
	ret = (fwrite(&r, sizeof(_eReal), 1, f->getFileHandle()) == 1)
			? FileError :: success
			: FileError :: writeError;
}

void Environment :: write(const FileRef* f, const _eByte b, FileError :: _eEnum &ret)
{
	_mSchema(write);
	fileWriteFlush(f);
	const int temp(static_cast<int>(b));
	ret = (putc(temp, f->getFileHandle()) == temp)
			? FileError :: success
			: FileError :: writeError;
}

//--------------------------------------------------------------------------
// Function: Environment :: scan(const FileRef* f, <type> &, FileError &ret)
// Purpose:  Read the specified character type from file.
//--------------------------------------------------------------------------
void Environment :: scan(const FileRef* f, _rstring &s, const _eInt len, FileError :: _eEnum &ret)
{
	_mSchema(scan);
	fileReadFlush(f);
	FILE * const file = f->getFileHandle();
	_eCstring temp(len);		// allocate buffer required for the c-string
	size_t charsRead = fread(temp.changeStr(), sizeof(_eChar), len, file);
	ret = (charsRead == static_cast<size_t>(len))
			? FileError :: success
			: feof(file)
				? FileError :: endOfFile
				: (ferror(file) && errno == EBADF)
					? FileError :: attribError 				// file wasn't opened for reading.
		  			: FileError :: readError;				// other miscellaneous error.
	s = _eSeq<_eChar>(charsRead, temp.str());
}

void Environment :: scan(const FileRef* f, _eChar &c, FileError :: _eEnum &ret)
{
	_mSchema(scan);
	fileReadFlush(f);
	FILE * const file = f->getFileHandle();
	int ch = getc(file);
	if (ch == EOF)
	{
		ret = 	  feof(file)
				? FileError :: endOfFile					// reached end of file
		  		: (ferror(file) && (errno == EBADF))
					? FileError :: attribError				// file wasn't opened for reading.
					: FileError :: readError;				// other miscellaneous error.
	}
	else
	{
		ret = FileError :: success;
		c = static_cast<_eNativeChar>(ch);
	}
}

//--------------------------------------------------------------------------
// Read a byte from file.
//--------------------------------------------------------------------------
void Environment :: read(const FileRef* f, _eByte &b, FileError :: _eEnum &ret)
{
	_mSchema(read);
	fileReadFlush(f);
	FILE * const file = f->getFileHandle();
	int ch = getc(file);
	if (ch == EOF)
	{
		ret = feof(file)
				? FileError :: endOfFile					// reached end of file
		  		: (ferror(file) && (errno == EBADF))
					? FileError :: attribError				// file wasn't opened for reading.
					: FileError :: readError;				// other miscellaneous error.
	}
	else
	{
		ret = FileError :: success;
		b = static_cast<_eByte>(ch);
	}
}

// Read up to 'numBytes' of data from file, returning it in 's'
void Environment :: read(const FileRef* f, _eSeq<_eByte> &s, const _eInt numBytes, FileError :: _eEnum &ret)
{
	_mSchema(read);

	fileReadFlush(f);													// flush data written if we are reading
	FILE * const file = f->getFileHandle();
	const int localBufferSize = 128;
	_eByte localBuffer[localBufferSize];								// local buffer to use of numBytes is small enough
	const bool alloc = (numBytes > localBufferSize);					// true if we need to allocate a buffer

	_eByte *temp = (alloc)
					? static_cast<_eByte*>(::_eMem::alloc(static_cast<size_t>(numBytes) * sizeof(_eByte)))
																	 	// allocate a buffer of appropriate size to hold the data
					: localBuffer;

	size_t bytesRead = fread(temp, sizeof(_eByte), static_cast<size_t>(numBytes), file);
																		// read in the data to the buffer
	// Set the return code as appropriate
	ret = (bytesRead == static_cast<size_t>(numBytes))
			? FileError :: success
			: feof(file) ? FileError :: endOfFile
				: (ferror(file) && errno == EBADF)
					? FileError :: attribError							// file wasn't opened for reading
					: FileError :: readError;							// other miscellaneous error.

	s = _eSeq<_eByte>();												// default the return to empty

	// If the read returned something valid, build the byte-sequence that is the result
	if (bytesRead <= static_cast<size_t>(numBytes))						// note: if bytesRead is negative, the loop will terminate immediately
	{
		for (size_t i = 0; i < bytesRead; i++)
		{
			s = s.append(temp[i]);
		}
	}

	if (alloc)
	{
		_eMem::free(temp, static_cast<size_t>(numBytes) * sizeof(_eByte));	// release the buffer
	}
}

// This reads integers in big-endian (MS Byte first) format.
// Keep this formatting matching that of the print for integers (see earlier in this file).
// Note: Result value is undefined if an error occures (i.e. if ret != Success).
void Environment :: read(const FileRef* f, _eInt &n, const _eInt numBytes, FileError :: _eEnum &ret)
{
	_mSchema(read);
	_mBeginPre _mCheckPre(numBytes > 0, "0,0");
	fileReadFlush(f);
	FILE * const file = f->getFileHandle();
	const size_t maxBytes = 2 * sizeof(_eInt);							// the maximum bytes in any integer we are prepared to handle
	_eByte buffer[maxBytes];

	size_t bytesToRead = static_cast<size_t>(numBytes);
	bool ok;
	if (bytesToRead > maxBytes)
	{
		// We have been asked to read more bytes than the size of our buffer, so skip the extra bytes.
		bytesToRead = maxBytes;
#if defined(_Win64)
		ok = (_fseeki64(file, static_cast<size_t>(numBytes) - bytesToRead, SEEK_CUR) == 0);
#else
		ok = (fseek(file, static_cast<long>(static_cast<size_t>(numBytes) - bytesToRead), SEEK_CUR) == 0);
#endif
	}
	else
	{
		ok = true;
	}

	if (ok)
	{
		ok = (fread(buffer, bytesToRead, 1, file) == 1);			// read in the data to the buffer and check one object read
	}

	if (ok)
	{
		// Get the first byte and sign-extend it to an integer
		n = static_cast<signed char>(buffer[0]);
		for (size_t i = 1; i < bytesToRead; ++i)
		{
			n = (n << 8) | buffer[i];
		}
		ret = FileError :: success;
	}
	else
	{
		ret = feof(file)
				? FileError :: endOfFile
				: (ferror(file) && errno == EBADF)
					? FileError :: attribError						// file wasn't opened for reading
					: FileError :: readError;						// other miscellaneous error
	}
}

void Environment :: read(const FileRef* f, _eInt &n, FileError :: _eEnum &ret)
{
	_mSchema(read);
	fileReadFlush(f);
	FILE * const file = f->getFileHandle();
	ret = (fread(&n, sizeof(_eInt), 1, file) == 1)
			? FileError :: success
		  	: feof(file)
				? FileError :: endOfFile
		  		: (ferror(file) && errno == EBADF)
		  			? FileError :: attribError						// file wasn't opened for reading.
		  			: FileError :: readError;						// other miscellaneous error.
}

void Environment :: read(const FileRef* f, _eReal &r, FileError :: _eEnum &ret)
{
	_mSchema(read);
	fileReadFlush(f);
	FILE * const file = f->getFileHandle();
	ret = (fread(&r, sizeof(_eReal), 1, file) == 1)
			? FileError :: success
			: feof(file)
				? FileError :: endOfFile
				: (ferror(file) && errno == EBADF)
					? FileError :: attribError						// file wasn't opened for reading.
		  			: FileError :: readError;						// other miscellaneous error.
}


//--------------------------------------------------------------------
// Function: Environment :: getImagePath() const
// Purpose:  Return the path, ending in a backslash, to the executable
//           image that is currently executing.
//--------------------------------------------------------------------
_rstring Environment :: getImagePath() const
{
	return ImagePath;
}

//--------------------------------------------------------
// Function: Environment :: getCurrentDirectory()
// Purpose:  Get the current default drive and directory
// Return:   A union of _eString with void.
//--------------------------------------------------------
_eUnion Environment :: getCurrentDirectory() const
{
	_eNativeChar buffer[NAME_MAX + 1];				// 1 extra character for a trailing "\"
	_eNativeChar osPathSep = _eosInterface :: getPathSeparator();
	if (_mGetcwd(buffer, NAME_MAX)==NULL)
	{
		return _eUnion(_mNullPtr(_eAny));		// returning null for failure
	}
	// ensure there is a trailing path-separator in the path
	size_t len = _mStrlen(buffer);
	if (buffer[len-1] != osPathSep)
	{
		buffer[len] = osPathSep;
		buffer[len+1] = _mCstr('\0');
	}
	return _eUnion(_mWrapStorable(_lString(buffer), _rstring));
}

//-------------------------------------------------------------------
// Function: Environment :: setCurrentDirectory(const _rstring path)
// Purpose:  Set the current default drive and directory
//-------------------------------------------------------------------
void Environment :: setCurrentDirectory(const _rstring path, FileError :: _eEnum & err)
{
	if (_mChdir(_eCstring(path).str()) == 0)
	{
		err = FileError :: success;
	}
	else
	{	// The call failed, so the default directory will not have been changed. We decide which FileError to
		// return here. Note that POSIX defines more errors than we currently deal with, as follows:
		// ENAMETOOLONG	if the path length exceeds the system maximum (as defined by global 'NAME_MAX')
		// ENOTDIR		if a component of the path is not a directory
		// In these cases we just return otherError ...
		err = (errno == ENOENT)
				? FileError :: directoryNotFound
				: errno == EACCES
					? FileError::permError
					: FileError::otherError;
	}
}

//--------------------------------------------------------------------------------------------------
// Function: Environment :: normalizeFile(const _rstring defaultPath, const _rstring fileName) const
// Purpose:  Given a filename and a default path ("defaultPath"), split it into a full path and
//           a filename ("fileName") without path. The default path must either be empty (in which
//           case the current directory is assumed) or end in "\". The split parts are held in the
//           FilePath handle object returned on success, else null-handle is returned.
//--------------------------------------------------------------------------------------------------
_eHndlUnionVoid<FilePath> Environment :: normalizeFile(const _rstring defaultPath, const _rstring fileName) const
{
    #if !defined(NDEBUG)
    _mBeginPre _mCheckPre (((defaultPath._oHash () == 0) || (_eosInterface :: getPathSeparator () == defaultPath.last ())), "0,0");
    #endif
	const _eCstring firstpart(defaultPath);
	const _eCstring otherpart(fileName);
	return _eosInterface :: normalizefilename(firstpart.str(), otherpart.str());
}

//--------------------------------------------------------------------------
// Function: Environment :: caseSensitiveFileNames()
// Purpose:  Ask the OS whether or not it supports casesensitive filenames.
//--------------------------------------------------------------------------
_eBool Environment :: caseSensitiveFileNames()
{
	return _eosInterface :: caseSensitiveFileNames();
}

//--------------------------------------------------------------------------
// Function: Environment :: pathSeparator()
// Purpose:  Ask the OS what the path separator character is.
//--------------------------------------------------------------------------
_eChar Environment :: pathSeparator()
{
	return _eosInterface :: getPathSeparator();
}

//---------------------------------------------------------------------------------
// Function: Environment :: setCurrentThreadPriority(const _eNat priority)
// Purpose:  Set the priority of the current process and thread to that specified
//           (range is 1 to 20, where 1 is the lowest priority).
//---------------------------------------------------------------------------------
void Environment :: setCurrentThreadPriority(const _eNat priority)
{
	_mSchema(setCurrentThreadPriority);
    _mBeginPre _mCheckPre (((1 <= priority) && (priority <= 20)), "0,0");
	_eosInterface :: setCurrentThreadPriority(priority);
}

//-------------------------------------------------------------------------------------------
// Function: Environment :: execute( const _rstring command, const _eSeq < _rstring > args,
//                                   const FileRef *stdinfile, const FileRef *stdoutfile,
//                                   const FileRef *stderrfile, FileError :: _eEnum & err )
// Purpose:  Execute the specified command under the local OS ...
//-------------------------------------------------------------------------------------------

// This schema has been moved to file 'CommandExecutor.cpp' for the purposes of efficient
// linking - if the schema is not referenced in user code, then module 'CommandExecutor'
// will not get linked in ...

//----------------------------------------------------------------------------
// Function: Environment :: clock() const
// Purpose:  Obtain the amount of processor time used by the calling process.
//           The 'clocksPerSecond' function will return the units used.
//----------------------------------------------------------------------------

// This function has been moved to NTSystemCalls and LinuxSystemCalls because its implementation
// is operating system dependant

//-----------------------------------------------------------------------------------------
// Function: Environment :: clocksPerSecond() const
// Purpose:  Obtain the units (as a fraction of a second) returned by the 'clock' function.
//-----------------------------------------------------------------------------------------

// This function has been moved to NTSystemCalls and LinuxSystemCalls because its implementation
// is operating system dependant

//-----------------------------------------------------------------------------
// Function: Environment :: getCurrentDateTime()
// Purpose:  Get the current date and time, in binary format that a developer
//           can convert to his required string format as required.
// Notes:	 The time returns in in coordinated-universal-time (UTC) format.
//-----------------------------------------------------------------------------
_eHndl < Time > Environment :: getCurrentDateTime() const
{
	time_t theTime;
	struct tm *timeTemp;
	time(&theTime);
	timeTemp = gmtime(&theTime);		// Converts the time value (time_t) to a struct tm*
	return _eHndl < Time >(new Time(timeTemp->tm_sec, timeTemp->tm_min, timeTemp->tm_hour, timeTemp->tm_wday, timeTemp->tm_mday, timeTemp->tm_mon + 1, timeTemp->tm_year + 1900));
}

//------------------------------------------------------------------------------
// Function: getEnvironmentVar(_rstring varName)
// Purpose:  Get the value associated with the specified environment variable
//           name. If the variable is not defined in the current environment,
//           return null.
//------------------------------------------------------------------------------
_eUnion Environment :: getEnvironmentVar(const _rstring varName) const
{
	const _eCstring vn(varName);
#if defined(_dUnicode)
  #if defined(_MSC_VER)
	_eNativeChar *envVar = _wgetenv(vn.str());
  #else
	// We don't have a wide character version of getenv in glibc and we can't write a thread-safe one, so do the conversion here
	char* nvarname;
	wcs2ncs(nvarname, vn.str());
	_eNativeChar* envVar = NULL;
	const char* var = getenv(nvarname);
	if (var != NULL)
	{
		ncs2wcs(envVar, var);
	}
  #endif
#else
	_eNativeChar *envVar = getenv(vn.str());
#endif
    return (envVar == NULL)
			? _eUnion (_mNull)
			: _eUnion (_mWrapStorable (_lString(envVar), _eSeq < _eChar >));
}

//------------------------------------------------------------------------------
// Function: Environment :: getImageVersion()
// Purpose:  Get the version information for an executable image (this info
//           will have been stored in the 'ProductVersion' field of the version
//           resource of the project) as a sequence of integers (4 integers for
//           Win32). The argument represents the name of the module we want to
//           query. If this is an empty string, then the module queried is the
//           currently running executable, else the name of another currently
//           loaded module (loaded into the currently running executable's
//           address space, that is) can be passed, for example the name of a
//           DLL such as "msvcrt.dll" ...
//------------------------------------------------------------------------------
_eSeq < _eInt > Environment :: getImageVersion(const _rstring moduleName) const
{
	const _eCstring mn(moduleName);
	const _eNativeChar *pmn = mn.str();
	return _eosInterface :: getImageVersion(_mStrlen(pmn) > 0 ? pmn : 0);
}

//------------------------------------------------------------------------------
// Function: Environment :: getMemoryUsed()
// Purpose:  Return the total number of bytes allocated by the runtime.
//------------------------------------------------------------------------------
_eNat Environment :: getMemoryUsed() const
{
	_eSize temp = _eMem::getManagedTotalAllocated() + _eMem::getUnmanagedTotalAllocated() - _eMem::getTotalInFreelists();
	// The memory used may be more than we can represent in an _eInt, in which case we need to truncate it.
	// Also, if _eSize ends up as a signed type, it may wrap round and appear to be negative.
	return (temp > static_cast<_eSize>(_lMaxInt) || static_cast<_eNat>(temp) < 0) ? _lMaxInt : static_cast<_eNat>(temp);
}

//---------------------------------------------------------------------------------------------
// Function: Environment :: whichOs()
// Purpose:  Determine which operating system is currently running. Return details of OS-type
//           (windows95, 98, NT etc) and major and minor OS-version numbers ...
//---------------------------------------------------------------------------------------------
_eHndl<OsInfo> Environment :: getOsInfo() const
{
	return _eosInterface :: getOsInfo();
}

//
// Functions to allow the setting of the global runtime-check-enabling variables via the Environment
//

void Environment :: setRuntimeOptionState (const DebugType :: _eEnum debOpt, const _eBool state)
{
    _mSchema (setRuntimeOptionState);
	switch(debOpt)
	{
		case DebugType :: preConditions:
		{
			_eEscherRuntime::EnablePre = state;
			break;
		}
		case DebugType :: postConditions:
		{
			_eEscherRuntime::EnablePost = state;
			break;
		}
		case DebugType :: loopInvariants:
		{
			_eEscherRuntime::EnableLoopInvariant = state;
			break;
		}
		case DebugType :: loopVariants:
		{
			_eEscherRuntime::EnableLoopDecrease = state;
			break;
		}
		case DebugType :: specVariants:
		{
			_eEscherRuntime::EnableSpecDecrease = state;
			break;
		}
		case DebugType :: impVariants:
		{
			_eEscherRuntime::EnableImpDecrease = state;
			break;
		}
		case DebugType :: embAsserts:
		{
			_eEscherRuntime::EnableEmbeddedAssert = state;
			break;
		}
		case DebugType :: postAsserts:
		{
			_eEscherRuntime::EnablePostAssert = state;
			break;
		}
		case DebugType :: lastChoices:
		{
			_eEscherRuntime::EnableLastChoice = state;
			break;
		}
		case DebugType :: classInvariants:
		{
			_eEscherRuntime::EnableClassInvariant = state;
			break;
		}
		case DebugType :: constraints:
		{
			_eEscherRuntime::EnableConstraints = state;
			break;
		}
	}
}

void Environment :: setRuntimeOptionsState (const _eSet < DebugType :: _eEnum > debOpts, const _eBool state)
{
    _mSchema (setRuntimeOptionsState);
    {
        const _eInt _vCapture_24_74 = debOpts._oHash ();
        _eInt _vLoopCounter_24_71;
        _vLoopCounter_24_71 = 0;
        for (;;)
        {
            if ((_vLoopCounter_24_71 == _vCapture_24_74)) break;
            setRuntimeOptionState (debOpts [_vLoopCounter_24_71], state);
            _vLoopCounter_24_71 = _oSucc (_vLoopCounter_24_71);
        }
    }
}

void Environment :: setRuntimeOption (const DebugType :: _eEnum debOpt)
{
    _mSchema (setRuntimeOption);
    setRuntimeOptionState (debOpt, true);
}

void Environment :: setRuntimeOptions (const _eSet < DebugType :: _eEnum > debOpts)
{
    _mSchema (setRuntimeOptions);
    {
        const _eInt _vCapture_22_74 = debOpts._oHash ();
        _eInt _vLoopCounter_22_71;
        _vLoopCounter_22_71 = 0;
        for (;;)
        {
            if ((_vLoopCounter_22_71 == _vCapture_22_74)) break;
            setRuntimeOption (debOpts [_vLoopCounter_22_71]);
            _vLoopCounter_22_71 = _oSucc (_vLoopCounter_22_71);
        }
    }
}

void Environment :: clrRuntimeOption (const DebugType :: _eEnum debOpt)
{
    _mSchema (clrRuntimeOption);
    setRuntimeOptionState (debOpt, false);
}

void Environment :: clrRuntimeOptions (const _eSet < DebugType :: _eEnum > debOpts)
{
    _mSchema (clrRuntimeOptions);
    {
        const _eInt _vCapture_24_74 = debOpts._oHash ();
        _eInt _vLoopCounter_24_71;
        _vLoopCounter_24_71 = 0;
        for (;;)
        {
            if ((_vLoopCounter_24_71 == _vCapture_24_74)) break;
            clrRuntimeOption (debOpts [_vLoopCounter_24_71]);
            _vLoopCounter_24_71 = _oSucc (_vLoopCounter_24_71);
        }
    }
}

void Environment :: setMaxCheckNestLevel (const _eNat level)
{
	_eEscherRuntime::MaxCheckNesting = static_cast<unsigned int>(level);
}

void Environment :: startProfiling()
{
	_mSchema(startProfiling);
#if _dProfile
	_eCallStack :: startProfiling();
#endif
}

// Write profile info to an already-open and writable file
void Environment :: profile(const FileRef* f, FileError :: _eEnum &ret)
{
	_mSchema(profile);

#if _dProfile
	_eCallStack :: stopProfiling();
	const _eCpuTimer totalProfilingTime = _eCallStack :: getTotalProfilingTime();
	const _eCpuTimer tenThousand = static_cast<_eCpuTimer>(10000);

	if (totalProfilingTime < tenThousand)
	{
		ret = (fprintf
				(	f->getFileHandle(),
					"Unable to profile (total profiling time of %d is too low)\n",
					static_cast<unsigned int>(totalProfilingTime)
				) > 0)
				? FileError :: success : FileError :: writeError;
	}
	else
	{
		const _eCpuTimer quantum = totalProfilingTime/tenThousand;
		FILE * const hndl = f->getFileHandle();

		_eMethod const * p;
		ret = FileError :: success;
		for (p = _eEscherRuntime::listOfMethods; p != NULL && ret == FileError :: success; p = p->next)
		{
			if (p->writeProfileInfo(hndl, quantum) <= 0)
				ret = FileError :: writeError;
		}
	}
#else
	ret = (fprintf(f->getFileHandle(), "This program was not built with profiling enabled\n") > 0)
			? FileError :: success : FileError :: writeError;
#endif

}

_eHndl < _eInstblTypeInfo > Environment :: _aMyTypeInfo ()
{
    static _eHndl<_eInstblTypeInfo> ti;
    ti = _eHndl<_eInstblTypeInfo>(new _eInstblTypeInfo(_estcBasicTypeCode :: BUILTIN_TYPEIDX_ENVIRONMENT, false));
    return ti;
}

_eHndl < _eInstblTypeInfo > FileRef :: _aMyTypeInfo ()
{
    static _eHndl<_eInstblTypeInfo> ti;
    ti = _eHndl<_eInstblTypeInfo>(new _eInstblTypeInfo(_estcBasicTypeCode :: BUILTIN_TYPEIDX_FILEREF, false));
    return ti;
}

// End.
