// **************************************************************************
// * File:   LowLevelSupport.cpp											*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Code to implement low-level interface to the local OS			*
// *         environment (ie. the local filesystem). All primitives used in *
// *         the code are POSIX complient, and where necessary we provide	*
// *         macros to vendor specific calls.								*
// **************************************************************************

//---------
// Includes
//---------

#include "Ertsys.hpp"
#include "CharHelpers.hpp"

// File inclusions for full declarations

#include "LowLevelSupport_1.hpp"

#include <cerrno>			// for ENOENT etc.
#include <sys/stat.h>

// Includes and macros to provide POSIX compatability (the macros defined in the Microsoft section allow us to
// use the POSIX names for POSIX standard identifiers in the code)

#if defined(_MSC_VER)

  #include <direct.h>		// for '_getcwd' and '_mkdir'

  // Have to use a macro for mkdir because although windows defines it's main make-directory function as
  // '_mkdir', it also defines 'mkdir' that takes 1 parameter, making it incompatable with POSIX (GRRRR!!!!)
 #if defined(_dUnicode)
  #define _mMkdir(file, flags) _wmkdir(file)
 #else
  #define _mMkdir(file, flags) _mkdir(file)
 #endif

  #define S_ISDIR(mode) (mode & _S_IFDIR)
  #define S_IWUSR _S_IWRITE
  #define S_IRUSR _S_IREAD
  #define S_IXUSR 0		// dummy
  #define S_IRGRP 0		// dummy
  #define S_IWGRP 0		// dummy
  #define S_IXGRP 0		// dummy
  #define S_IROTH 0		// dummy
  #define S_IWOTH 0		// dummy
  #define S_IXOTH 0		// dummy

#elif defined(__GNUC__)

  // the POSIX includes for the 'mkdir' function
  #include <sys/types.h>
  #include <sys/stat.h>

  #define _stat stat

  #if defined(_dUnicode)
   #include "LinuxSystemCalls.hpp"		// for _wmkdir, _wstat
   #define _mMkdir(file, flags) _wmkdir(file, flags)
  #else
   #define _mMkdir(file, flags) mkdir(file, flags)
  #endif

#endif

// Schema to create a directory component
void _eLowLevelSupport :: mkdircomp (const _eSeq < _eChar > compname, FileError :: _eEnum & err)
{
    _mSchema (mkdircomp);
	_eCstring cCompname(compname);

	// we will first see if the directory already exists - in this case we will flag success...
	struct _stat buffer;				// stat's result holder
	if (_mStat(cCompname.str(), &buffer) == 0)
	{
		// Either a file or directory of the given name exists - if it is a directory we have succeeded,
		// if it is a file we return 'createError' which indicates the directory could not be created for this reason.
		err = (S_ISDIR(buffer.st_mode)) ? FileError :: success : FileError :: createError;
		return;
	}

	// is not already a directory, so try to create it and see what happens...
	// note that errno == EEXIST will occur in the case we try to create C: or similar - this is also a success.
	if (_mMkdir(cCompname.str(), S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IWGRP | S_IXGRP | S_IROTH | S_IWOTH | S_IXOTH) != 0 && errno != EEXIST)
	{
		err = (errno == EACCES || errno == EROFS) ? FileError :: permError :	// Permissions insufficient - root dir or drive not writable by user
			errno == ENOSPC ? FileError :: diskFull :		// Insufficient space available
			FileError :: otherError;						// General catchall for other miscellaneous errors that may occur
		return;
	}
	err = FileError :: success;
}

// See bug report 726 for comments on this method ...
_eSeq < _eByte > _eLowLevelSupport :: realToBytes (const _eReal num)
{
    _mFunction (realToBytes);
	const unsigned char *tmp = reinterpret_cast<const unsigned char *>(&num);
    return _eSeq < _eByte > ()._lAppend (static_cast < _eByte > (*(tmp+0)), static_cast < _eByte > (*(tmp+1)),
        static_cast < _eByte > (*(tmp+2)), static_cast < _eByte > (*(tmp+3)), static_cast < _eByte > (*(tmp+4)),
        static_cast < _eByte > (*(tmp+5)), static_cast < _eByte > (*(tmp+6)), static_cast < _eByte > (*(tmp+7)));
}

// See bug report 726 for comments on this method ...
_eReal _eLowLevelSupport :: bytesToReal (const _eSeq < _eByte > bytes)
{
    _mFunction (bytesToReal);
	unsigned char tmp[8];
	for (int i = 0; i < 8; i++)
	{
		tmp[i] = bytes[i];
	}
	return *(reinterpret_cast<_eReal *>(tmp));
}

// Convert an integer to a sequence of bytes
_eSeq < _eByte > _eLowLevelSupport :: intToBytes (_eInt n, _eNat numBytes)
{
    _mFunction (intToBytes);
	_eByte* temp = new _eByte[numBytes];
	int count = static_cast<int>(numBytes);
	const int limit = (count > static_cast<int>(sizeof(_eInt))) ? count - static_cast<int>(sizeof(_eInt)) : 0;
	for (int i = 0; i < limit; ++i)
	{
		temp[i] = (n < 0) ? static_cast<_eByte>(255) : static_cast<_eByte>(0);
	}

	do
	{
		temp[--count] = static_cast<_eByte>(static_cast<unsigned int>(n) & 255);
		n >>= 8;				// right shift of signed value is safe here because we don't use the shifted-in bits
	} while (count != limit);

	_eSeq < _eByte > rslt (numBytes, temp);
	delete temp;
	return rslt;
}

_eInt _eLowLevelSupport :: bytesToInt (const _eSeq < _eByte > bytes)
{
    _mFunction (bytesToInt);
	_eInt rslt = static_cast<_eInt>(static_cast<signed char>(bytes[0]));	// convert first byte to _eInt with sign extension
	const int len = static_cast<int>(bytes._oHash());
	for (int i = 1; i < len; ++i)
	{
		rslt = (rslt << 8) | static_cast<unsigned char>(bytes[i]);
	}
	return rslt;
}

_eHndl < _eInstblTypeInfo > _eLowLevelSupport :: _aMyTypeInfo ()
{
    static _eHndl<_eInstblTypeInfo> ti;
    ti = _eHndl<_eInstblTypeInfo>(new _eInstblTypeInfo(_estcBasicTypeCode :: BUILTIN_TYPEIDX_LOWLEVELSUPPORT, false));
    return ti;
}

// End of file.
