// **************************************************************************
// * File:   Callstack.cpp										   			*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Definition of global call-stack for runtime de-bugging (call-	*
// *         stack is displayed after an exception).						*
// **************************************************************************

//---------
// Includes
//---------

#include "Ertsys.hpp"

#include <cstdio>		// for snprintf
#include <cwchar>		// for wcslen etc.
#include "CharHelpers.hpp"

//-------------------------------------
// Code - just initialise stack to null
//-------------------------------------

namespace _eEscherRuntime
{
	_eCallStack * threadCallStack = NULL;
	_eMethod const * listOfMethods = NULL;
}

// Global _tSelf for use in constructor inherits and member initialisation code
// Only needed if _dCallStack is defined in some compiled code; but we declare it unconditionally here so that a runtime
// library built with _dCallStack turned off can be used with an application compiled with _dCallStack on
_eMethod _tSelf(_mCstr("bld"), 0, 0);

//---------------------------
// Formatted output function
//---------------------------

_eNativeChar const * _eMethod::formattedStr() const
{
	const _eNativeChar formatString[] =
#if defined(_dUnicode)
		L"%ls (%ls, line %u)";
#else
		"%s (%s, line %u)";
#endif

    // Allocate a buffer for the string we're going to store
	const size_t sizeNeeded =
				sizeof(formatString)/sizeof(_eNativeChar)  		// length of the string
				+ _mStrlen(functionname) - 2					// plus length of the function name less length of "%s"
				+ _mStrlen(filename) - 2					    // plus length of the filename less length of "%s"
				+ _lDigitsMax - 2								// plus max length of line number less length of "%u"
				+ 1; 											// plus null terminator
    _eNativeChar * buffer = new _eNativeChar[sizeNeeded];
	_mSnprintf(buffer, sizeNeeded, formatString, functionname, filename, linenumber);
    return buffer;
}

#if _dProfile

// Write a line to the profile file
// Parameter 'quantum' is the time quantum we are using, which is the total execution time divided by ten thousand,
// i.e. each time quantum represents 0.01% of the total execution time
int _eMethod::writeProfileInfo(FILE *f, _eCpuTimer quantum) const
{
	const unsigned int totalTime = static_cast<unsigned int>(timeInMethod/quantum);
	const unsigned int methodOnlyTime = static_cast<unsigned int>(timeReallyInMethod/quantum);
	return (totalTime == 0)
			? 1
			// Print: methodname method+childrenTime methodTime file(line)
			: _mFprintf(	f,
#if defined(_dUnicode)
							L"%-32.32ls %3d.%02d %3d.%02d  %ls(%d)\n",
#else
							"%-32.32s %3d.%02d %3d.%02d  %s(%d)\n",
#endif
							functionname,
							totalTime/100, totalTime % 100,
							methodOnlyTime/100, methodOnlyTime % 100,
							filename, linenumber
					 );
}

// Flag to say whether profiling is enabled.
// This _must_ be initialised to false, as we cannot rely on the global _tSelf being initialised
// during initialisation of constants (i.e. before the start of 'main'), so dereferencing 'method'
// could cause a crash.
bool _eCallStack::profilingEnabled = false;

// This doesn't need to be initialised, but we will do so anyway in case static analysers object if we don't
_eCpuTimer _eCallStack::profileStartTime = 0;

// This does need to be initialised to zero
_eCpuTimer _eCallStack::totalProfilingTime = 0;

// Begin a method call
void _eCallStack::beginCall(_eCpuTimer now)
{
	timeOfCall = now;
	totalBeforeCall = method->timeInMethod;
	timeInChildren = 0;						// I'm assuming this will get converted to a 64-bit zero
}

// This method is called by the constructor of _eCallStack if profiling is enabled
// Equivalent to: beginCall(_lGetCpuTimer())
void _eCallStack::beginCall()
{
	timeOfCall = _lGetCpuTimer();
	totalBeforeCall = method->timeInMethod;
	timeInChildren = 0;						// I'm assuming this will get converted to a 64-bit zero
}

// End a method call
void _eCallStack::endCall(_eCpuTimer now)
{
	const _eCpuTimer timeInCall = now - timeOfCall;
	method->timeInMethod = totalBeforeCall + timeInCall;
	if (next != NULL)
	{
		next->timeInChildren += timeInCall;
	}
	method->timeReallyInMethod += timeInCall - timeInChildren;
}

// The following method is called by the destructor of _eCallStack if profiling is enabled
// Equivalent to: endCall(_lGetCpuTimer())
void _eCallStack::endCall()
{
	const _eCpuTimer timeInCall = _lGetCpuTimer() - timeOfCall;
	method->timeInMethod = totalBeforeCall + timeInCall;
	if (next != NULL)
	{
		next->timeInChildren += timeInCall;
	}
	method->timeReallyInMethod += timeInCall - timeInChildren;
}

// Start the profiling counters
void _eCallStack::startProfiling()
{
	if (!profilingEnabled)
	{
		const _eCpuTimer now = _lGetCpuTimer();
		for (_eCallStack* ptr = _eEscherRuntime::threadCallStack; ptr != NULL; ptr = ptr->next)
		{
			ptr->beginCall(now);
		}
		profilingEnabled = true;
		profileStartTime = now;
	}
}

// Stop the profiling counters, returning the total time since profiling was started
void _eCallStack::stopProfiling()
{
	if (profilingEnabled)
	{
		const _eCpuTimer now = _lGetCpuTimer();

		for (_eCallStack* ptr = _eEscherRuntime::threadCallStack; ptr != NULL; ptr = ptr->next)
		{
			ptr->endCall(now);
		}
		profilingEnabled = false;
		totalProfilingTime += (now - profileStartTime);
	}
}

#endif

// End.
