// **************************************************************************
// * File:   Casts.hpp										   				*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Definition of macros related to type checking and type casting	*
// **************************************************************************

#if !defined(_h_casts)
#define _h_casts 1

//-----------------------------------------------------------
// forward declaration needed by the xxxx[Non]Storable macros
//-----------------------------------------------------------

template<class X> class _eWrapperBase;
template<class X> class _eStorableWrapper;
template<class X> class _eBasicWrapper;
template<class X> class _eNonStorableWrapper;
template<class X, class Y> class _eEnumWrapper;
template<class X, class Y> class _eNonStorableEnumWrapper;

//---------------------------------------
// type test macros used for unions/froms
//---------------------------------------

#define _mWithin(pointer, type) \
			_lWithin(pointer, typeid(type))

#define _mWithinFrom(pointer, type) \
			(dynamic_cast<const type*>(pointer) != NULL)

#define _mWithinNonHandle(union_ptr, type) \
			_lWithin(union_ptr, typeid(type))

#define _mWithinRef(uni, type) 					\
			_lWithin(uni, typeid(_eRef< _eHndl< type > >))

#define _mWithinRefFrom(uni, type) 				\
			_lWithin(uni, typeid(_eRef< _eFrom< type > >))

#define _mWithinRefNonHandle(union_ptr, type) \
			_lWithin(union_ptr, typeid(_eRef< type >))

//---------------------------------
// macros to perform widening casts
//---------------------------------

#define _mAsFrom(pointer_X, class_Y_ancestor_of_X) \
			static_cast<const class_Y_ancestor_of_X*>(pointer_X)

#define _mAsFromChange(pointer_X, class_Y_ancestor_of_X) \
			static_cast<class_Y_ancestor_of_X*>(pointer_X)

//----------------------------------------
// wrapping of non-handle types for unions
//----------------------------------------

// create wrapper for storable non-handle class
#define _mWrapStorable(value, type) \
			static_cast<_eStorable *>(new _eStorableWrapper<type >(value))

// create wrapper for basic type
#define _mWrapBasic(value, type) \
			static_cast<_eStorable *>(new _eBasicTypeWrapper<type >(value))

// create wrapper for non-storable non-handle class
#define _mWrapNonStorable(value, type) \
			static_cast<_eAny *>(new _eNonStorableWrapper<type >(value))

#define _mWrapEnum(value, type) \
			static_cast<_eAny *>(new type :: _eWrapper(value))

#define _mWrapStorableEnum(value, type) \
			static_cast<_eStorable *>(new type :: _eWrapper(value))

//------------------
// Casting macros...
//------------------

#if _dCheckCast

  //******************
  //* ... with tests *
  //******************

  //--------------------------------
  // Casting macros for handle types
  //--------------------------------

  // handle types
  #define _mIs(pointer, type) \
  			  _lCheckType<type >(pointer, _mCurrentFile, __LINE__)
  #define _mIsChange(pointer, type) \
  			  _lCheckTypeChange<type >(pointer, _mCurrentFile, __LINE__)

  // version used during de-serialization
  #define _mIsForLoading(pointer, type, allowNull) \
  			  _lCheckTypeForLoading<type >(pointer, allowNull, _mCurrentFile, __LINE__)

  // ref's to storable/non-storable handle types
  #define _mIsRef(pointer, type) 			\
   			  _lCheckRefType<_eRef< _eHndl< type > > >(pointer, _mCurrentFile, __LINE__)
  #define _mIsRefChange(pointer, type) 		\
  			  _lCheckRefTypeChange<_eRef< _eHndl< type > > >(pointer, _mCurrentFile, __LINE__)

  //------------------------------
  // Casting macros for from types
  //------------------------------

  // from handle types
  #define _mIsFrom(pointer, type) \
  			  _lCheckNotNull(dynamic_cast<const type*>(pointer), _mCurrentFile, __LINE__)
  #define _mIsFromChange(pointer, type) \
 			  _lCheckNotNullChange(dynamic_cast<type*>(pointer), _mCurrentFile, __LINE__)

   // version used during de-serialization ...
  #define _mIsFromForLoading(pointer, type, allowNull) \
  			  _lCheckNotNullForLoading(dynamic_cast<const type*>(pointer), allowNull, _mCurrentFile, __LINE__)

  // ref's to from types
  #define _mIsRefFrom(pointer, type)			\
   			  (_lCheckNotNull(dynamic_cast<const _eWrapperBase<_eRef< _eFrom< type > > >*>(pointer), _mCurrentFile, __LINE__)->_lValue())
  #define _mIsRefFromChange(pointer, type)		\
 			  (_lCheckNotNullChange(dynamic_cast<_eWrapperBase<_eRef< _eFrom< type > > >*>(pointer), _mCurrentFile, __LINE__)->_lChangeValue()())

  //-------------------------------------
  // Casting macros for non-handle types
  //-------------------------------------

  // non-handle types
  #define _mIsNonHandle(union_ptr, type) \
  			  _lCheckNonHandle<type >(union_ptr, _mCurrentFile, __LINE__)
  #define _mIsNonHandleChange(union_ptr, type) \
  			  _lCheckNonHandleChange<type >(union_ptr, _mCurrentFile, __LINE__)

  // ref's to non-handle types
  #define _mIsRefNonHandle(union_ptr, type)			\
  			  _lCheckNonHandle<_eRef<type > >(union_ptr, _mCurrentFile, __LINE__)
  #define _mIsRefNonHandleChange(union_ptr, type)			\
  			  _lCheckNonHandleChange<_eRef<type > >(union_ptr, _mCurrentFile, __LINE__)

  //-------------------------------------------------
  // Casting macros with null checks for handle types
  //-------------------------------------------------

  #define _mIsNull(pointer) \
  			  _lCheckIsNull(pointer, _mCurrentFile, __LINE__)

  #define _mIsNotNull(pointer, type) \
  			  _lCheckNotNull<type >(pointer, _mCurrentFile, __LINE__)

  #define _mIsNotNullChange(pointer, type) \
  			  _lCheckNotNullChange<type>(pointer, _mCurrentFile, __LINE__)

  //----------------------------------------------
  // Casting macros with null checks for ref types
  //----------------------------------------------

  #define _mRefIsNull(ref) \
  			  ref.CheckIsNull(_mCurrentFile, __LINE__)

  #define _mRefIsNotNull(ref) \
  			  ref.CheckNotNull(_mCurrentFile, __LINE__)

  #define _mRefIsNotNullChange(ref) \
  			  ref.CheckNotNull(_mCurrentFile, __LINE__)

  //-------------------------------
  // Functions to support the above
  //-------------------------------

  inline void* _lCheckIsNull(const _eAnyBase* ptr, const _eNativeChar* file, const int line)
  {
  	if (ptr == NULL)
  	{
  		return NULL;
  	}
  	throw _xCannotHappen(_mCstr("'is' cast failed (value was not null)"), file, line);
  }

  template<class X> inline const X* _lCheckType(const _eAnyBase* ptr, const _eNativeChar* file, const int line)
  {
  	if (ptr != NULL && ptr->_lType() == typeid(X))
  	{
  		return static_cast<const X*>(ptr);
  	}
  	throw _xCannotHappen(_mCstr("'is' cast failed"), file, line);
  }

  // Version of '_lCheckType' used during de-serialization ...
  template<class X> inline const X* _lCheckTypeForLoading(const _eAnyBase* ptr, const bool allowNull, const _eNativeChar* file, const int line)
  {
  	if (ptr != NULL && ptr->_lType() == typeid(X))
  	{
  		return static_cast<const X*>(ptr);
  	}
	else if (ptr == NULL && allowNull)
  	{
  		return dynamic_cast<const X*>(ptr);		// casting (AnyBase*)NULL to (X*)NULL
  	}
  	throw _xCannotHappen(_mCstr("Unexpected type found in storage stream (exact-type)"), file, line);
  }

  template<class X> inline X* _lCheckTypeChange(_eAnyBase* ptr, const _eNativeChar* file, const int line)
  {
  	if (ptr != NULL && ptr->_lType() == typeid(X))
  	{
  		return static_cast<X*>(ptr);
  	}
  	throw _xCannotHappen(_mCstr("'is' cast failed"), file, line);
  }

  template<class X> inline X _lCheckRefType(const _eAnyBase* ptr, const _eNativeChar* file, const int line)
  {
  	if (ptr != NULL && ptr->_lType() == typeid(X))
  	{
  		return static_cast<const _eWrapperBase<X >*>(ptr)->_lValue();
  	}
  	throw _xCannotHappen(_mCstr("'is ref' cast failed"), file, line);
  }

  template<class X> inline X& _lCheckRefTypeChange(const _eAnyBase* ptr, const _eNativeChar* file, const int line)
  {
  	if (ptr != NULL && ptr->_lType() == typeid(X))
  	{
  		return static_cast<_eWrapperBase<X >*>(ptr)->_lChangeValue();
  	}
  	throw _xCannotHappen(_mCstr("'is ref' cast failed"), file, line);
  }

  template<class X> inline X _lCheckNonHandle(const _eAnyBase* ptr, const _eNativeChar* file, const int line)
  {
  	if (ptr != NULL && ptr->_lType() == typeid(X))
  	{
  		return (static_cast<const _eWrapperBase<X >* >(ptr))->_lValue();
  	}
  	throw _xCannotHappen(_mCstr("'is' cast failed (non-handle type)"), file, line);
  }

  template<class X> inline X& _lCheckNonHandleChange(_eAnyBase* ptr, const _eNativeChar* file, const int line)
  {
  	if (ptr != NULL && ptr->_lType() == typeid(X))
  	{
  		return (static_cast<const _eWrapperBase<X >* >(ptr))->_lChangeValue();
  	}
  	throw _xCannotHappen(_mCstr("'is' cast failed (non-handle type)"), file, line);
  }

  template<class X> inline const X* _lCheckNotNull(const X* ptr, const _eNativeChar* file, const int line)
  {
  	if (ptr != NULL)
  	{
  		return ptr;
  	}
  	throw _xCannotHappen(_mCstr("'is' cast failed (simple union was 'null')"), file, line);
  }

  // Version of '_lCheckNotNull' used during de-serialization
  template<class X> inline const X* _lCheckNotNullForLoading(const X* ptr, const bool allowNull, const _eNativeChar* file, const int line)
  {
  	if (ptr != NULL || allowNull)
  	{
  		return ptr;
  	}
  	throw _xCannotHappen(_mCstr("Unexpected type found in storage stream (from-type)"), file, line);
  }

  template<class X> inline X* _lCheckNotNullChange(X* ptr, const _eNativeChar* file, const int line)
  {
  	if (ptr != NULL)
  	{
  		return ptr;
  	}
  	throw _xCannotHappen(_mCstr("'is' cast failed (simple union was 'null')"), file, line);
  }

#else

  //-----------------
  // ...without tests
  //-----------------

  //--------------------------------------------------------------------------------------------
  // Casting macros for handle types ...
  // We use static casts here because the pointer could be null (or derrivitive of _eAnyBase *),
  // and this cast allows us to convert the null value to the target pointer type.
  //--------------------------------------------------------------------------------------------

  // handle types
  #define _mIs(pointer,type) \
  			  static_cast<const type*>(pointer)
  #define _mIsChange(pointer,type) \
  			  static_cast<type*>(pointer)

  // ref's to handle types
  #define _mIsRef(pointer, type) 						\
			  (static_cast<const _eWrapperBase<_eRef< _eHndl<type > > >*>(pointer)->_lValue())
  #define _mIsRefChange(pointer, type) 				\
			  (static_cast<_eWrapperBase<_eRef< _eHndl<type > > >*>(pointer)->_lChangeValue())

  //-------------------------------------------------------------
  // Casting macros for from types ...
  //-------------------------------------------------------------

  // from types
  #define _mIsFrom(pointer,type) \
  			  static_cast<const type*>(pointer)
  #define _mIsFromChange(pointer,type) \
  			  static_cast<type*>(pointer)

  // ref's to from types
  #define _mIsRefFrom(pointer, type)			\
   			  (static_cast<const _eWrapperBase<_eRef< _eFrom< type > > >*>(pointer)->_lValue())
  #define _mIsRefFromChange(pointer, type)		\
 			  (static_cast<_eWrapperBase<_eRef< _eFrom< type > > >*>(pointer)->_lChangeValue())

  //-------------------------------------
  // Casting macros for non-handle types
  //-------------------------------------

  // non-handle types
  #define _mIsNonHandle(union_ptr, type) \
			  (static_cast<const _eWrapperBase<type > *>(union_ptr)->_lValue())
  #define _mIsNonHandleChange(union_ptr, type) \
  			  (static_cast<_eWrapperBase<type > *>(union_ptr)->_lChangeValue())

  // ref's to non-handle types
  #define _mIsRefNonHandle(union_ptr, type)			\
			  (static_cast<const _eWrapperBase<_eRef<type > > *>(union_ptr)->_lValue())
  #define _mIsRefNonHandleChange(union_ptr, type)			\
  			  (static_cast<_eWrapperBase<_eRef<type > > *>(union_ptr)->_lChangeValue())

  //-----------------------------------------------------
  // Casting macros with null checks for handle types ...
  // Note there's no static casts here because we know
  // that the pointer is not null.
  //-----------------------------------------------------

  #define _mIsNull(pointer) 				NULL
  #define _mIsNotNull(pointer, type)		(pointer)
  #define _mIsNotNullChange(pointer, type)	(pointer)

  //----------------------------------------------
  // Casting macros with null checks for ref types
  //----------------------------------------------

  #define _mRefIsNull(ref)					NULL
  #define _mRefIsNotNull(ref)				(ref)
  #define _mRefIsNotNullChange(ref)			(ref)

#endif

//-------------------------------------------------------
// Casting macro for equality checks for non-handle types
//-------------------------------------------------------

#define _mEqualNonHandle(union_ptr, rhs, type) \
  (union_ptr != NULL && union_ptr->_lType() == typeid(type) && (static_cast<const _eWrapperBase<type >* >(union_ptr))->_lValue() == rhs)

//--------------------------------------------------------
// Global function for impementing "within" type check ...
//--------------------------------------------------------

inline bool _lWithin(const _eAnyBase *ptr, const std::type_info& type)
{
	return ptr != NULL && ptr->_lType() == type;
}

//--------------------------------------------------------------------
// Macro for implementing "like" type check
// The "ptr1 == ptr2" element accounts for the case of both being NULL
//--------------------------------------------------------------------

#define _mLike(ptr1, ptr2)	((ptr1 == ptr2) || (ptr1 != NULL && ptr2 != NULL && ptr1->_lType() == ptr2->_lType()))

//-----------------------------------------------------------------------
// Macro used to check for equality of base object when defining equality
// operator. Also checks whether the objects are the same to bypass
// comparisons of fields ...
//-----------------------------------------------------------------------

#define _mInheritEquality(type, name) \
	if (this == name) return true; \
	if (!(type :: operator == (static_cast<const type *>(name)))) return false;

//------------------------------------------------------------------------
// Macro used to check for equality of base object when defining equality
// operator for small classes ...
//------------------------------------------------------------------------

#define _mInheritSmallEquality(type, name)				\
	if (!(type :: operator == (static_cast<const type &>(name)))) return false;

//------------------------------------------------------------------------
// Macro used when defining equality operator and there is no base object,
// for a non-small class the same to bypass comparisons of fields ...
//------------------------------------------------------------------------

#define _mBeginEquality(name) \
	if (this == name) return true;

//------------------------------------------------------------------------
// Macro used when defining a member relative rank operator.
// Checks whether the two objects are the same to shorten the check.
//------------------------------------------------------------------------

#define _mBeginRank(name) \
	if (this == name) return _eRank :: same;

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
