// **************************************************************************
// * File:   NtSystemCalls.cpp											   	*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Code to implement an interface between the runtime and Win32	*
// * Note:   This interface was compiled and tested under Microsoft Visual  *
// *         C++ V6, and must be compiled with Microsoft language			*
// *         directives enabled (i.e. without the /Za switch)				*
// **************************************************************************

//---------
// Includes
//---------

#include "Ertsys.hpp"
#include "CharHelpers.hpp"

#if defined(_dUnicode)
#define UNICODE	(1)		// used by some of the MS header files
#endif

#include "NtSystemCalls.hpp"
#include <windows.h>
#include <winver.h>		// for version resource constants
#include <io.h>
#include <sys\stat.h>
#include <cerrno>
#include <ctime>		// for clock()

namespace _eosInterface
{
	// Static data
	static OSVERSIONINFOEX osvi;					// OS version info
	static int osVersionInfoType = 0;				// changed to 1 or 2 when OS version info is valid
	static bool canGetProcessTimes = false;			// changed to true if we are running under Windows NT/2000/XP
	static HANDLE currentProcess;					// handle or pseudo-handle to the current process

	// Function used by 'normalizefilename(..)' to expand possibly 'short' path components into long ones.
	_eHndlUnionVoid<FilePath> genNormalizedFilePathObject(const _rstring path, const _rstring fname);
	// Function used by 'genNormalizedFilePathObject(..)' - it provides functionality available under Windows2000, but not NT4.
	DWORD getLongPathName(const _eNativeChar *shortPath, _eNativeChar *longPath, DWORD maxBufLen);


	//---------------------------------------------------------------------------------
	// Function: _eosInterface :: Initialize()
	// Purpose: Set up any static data that is needed.
	//---------------------------------------------------------------------------------
	void initialize()
	{
		// Get the operating system type to make calls to clock() faster
		// Try calling GetVersionEx using the OSVERSIONINFOEX structure, which is supported on Windows NT versions 5.0 and later.
		// If that fails, try using the OSVERSIONINFO structure, which is supported on earlier versions of Windows and Windows NT

		ZeroMemory(&osvi, sizeof(OSVERSIONINFOEX));
		osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFOEX);

		if (GetVersionEx((OSVERSIONINFO*)&osvi))
		{
			osVersionInfoType = 2;				// we got the full info
		}
		else
		{
	   		// ok, OSVERSIONINFOEX didn't work, try OSVERSIONINFO ...
	   		osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	   		if (GetVersionEx((OSVERSIONINFO*)&osvi))
			{
				osVersionInfoType = 1;			// we got partial info
			}
			else
			{
				osVersionInfoType = 0;			// no info available
			}
		}

		// We can make the 'GetProcessTimes' system call if we are running under Windows NT 3.5 or later
		canGetProcessTimes = (osVersionInfoType != 0 && osvi.dwPlatformId == VER_PLATFORM_WIN32_NT);
		currentProcess = GetCurrentProcess();
	}

	//-----------------------------------------------------------------
	// Structure needed for version information extraction (not defined
	// in any Microsoft header - see MSDN for more information).
	//-----------------------------------------------------------------

	typedef struct
	{
		WORD  wLength;
		WORD  wValueLength;
		WORD  wType;
		WCHAR *szKey;
		WORD  *Padding1;
		VS_FIXEDFILEINFO Value;
	    WORD  *Padding2;
		WORD  *Children;
	} VS_VERSIONINFO;

	//---------------------------------------------------------------------------------
	// Function: _eosInterface :: normalizefilename(const _eNativeChar *path, const _eNativeChar *fname)
	// Purpose:  Given a filename and a default path ("path"), split it into a full
	//           path and a filename ("fname") without path. The default path must
	//           either be empty (in which case the current directory is assumed)
	//           or end in "\". The split parts are held in the FilePath handle object
	//           returned on success, else a null-handle is returned.
	//??? Note:  This could be drastically improved by checking whether the basepath
	//           ('path') actually ends in a backslash or not and processing appropriately
	//           depending on the result; currently, if the basepath doesn't end in a
	//           backslash (and the file doesn't specify a full-path), the logic will
	//           simply concatenate the path and file, resulting in a silly extension,
	//           eg. passing 'C:\Files\FredsFile.txt' as the path and 'BobsFile.txt'
	//           as the file, we would end up with the path as 'C:\Files\', the file
	//           as 'FredsFile' and the extension as 'txtBobsFile.txt'. It would be
	//           better if we allowed a basepath to not end in a backslash and then
	//           truncate it so that we only use the part upto the last backslash. The
	//           example above would then yield the file as 'BobsFile' and the extension
	//           as 'txt' which would be more useful!
	//---------------------------------------------------------------------------------
	_eHndlUnionVoid<FilePath> normalizefilename(const _eNativeChar *path, const _eNativeChar *fname)
	{
		_eNativeChar buffer[MAX_PATH];
		_eNativeChar *fileptr = NULL;
		_eNat getFullPathNameRes;

		if (fname[0] == '\\' || (fname[0] != '\0' && fname[1] == ':'))
		{
			// The filename includes a path from the root or a drive letter, so just normalise the filename
			getFullPathNameRes = GetFullPathName(fname, MAX_PATH, buffer, &fileptr);
		}
		else
		{
			// No root path, drive or machine name given so use the default path
			//??? This could to be improved - see note above.
			_eCstring tempresult(_mStrlen(path) + _mStrlen(fname) + 1);  // temp buffer
			_mStrcpy(tempresult.changeStr(), path);
			_mStrcat(tempresult.changeStr(), fname);
			getFullPathNameRes = GetFullPathName(tempresult.str(), MAX_PATH, buffer, &fileptr);
		}
		if (getFullPathNameRes == 0 || getFullPathNameRes >= MAX_PATH)	// if 'GetFullPathName(..)' failed or buffer not large enough..
		{
			return _eHndlUnionVoid<FilePath>();							// Returning null-handle for failure
		}
		// "buffer" now contains the full path and filename, and "fileptr" points to the final file component in it
		if (fileptr == NULL)
		{
			return genNormalizedFilePathObject(_lString(buffer), _mString(""));
		}
		else
		{
			const _rstring file = _lString(fileptr);
			*fileptr = '\0';
			return genNormalizedFilePathObject(_lString(buffer), file);
		}
	}

	//---------------------------------------------------------------------------------
	// Make chmod call to set file attributes
	//---------------------------------------------------------------------------------
	FileError :: _eEnum chmod(const _eNativeChar *fname, _eSet<FileAttribute :: _eEnum> atts)
	{
		int mode = atts._ovIn(FileAttribute :: read) ? _S_IREAD : 0;
		if (atts._ovIn(FileAttribute :: write))
		{
			mode = mode | _S_IWRITE;
		}
		int temp = _mChmod(fname, mode);
		return    temp == 0			? FileError :: success
				: errno == ENOENT 	? FileError :: fileNotFound
				: errno == EACCES 	? FileError :: permError
				: FileError :: otherError;
	}

	//-------------------------------------------------------------------
	// Function: _eosInterface :: caseSensitiveFileNames()
	// Purpose:  Obtain information from the OS as to whether or not the
	//           local filessystem is case sensitive.
	//-------------------------------------------------------------------
	bool caseSensitiveFileNames()
	{
		return false;				                        // file open under Windows is not case sensitive
	}

	//-------------------------------------------------------------------
	// Function: _eosInterface :: getPathSeparator()
	// Purpose:  Obtain information from the OS as to what the path
	//           separator character is on the local filessystem.
	//-------------------------------------------------------------------
	_eNativeChar getPathSeparator()
	{
		return _mCstr('\\');
	}

	//--------------------------------------------------------------------------------------------------------
	// Function: _eosInterface :: setCurrentThreadPriority(const _eNat priority)
	// Purpose:  Set the base priority of the current thread by setting the priority class of the current
	//           process, and the priority level of the current thread. 'Perfect' supports 20 priority levels.
	//           For more information, see MSDN entry "scheduling priorities".
	//--------------------------------------------------------------------------------------------------------

	static const _eMap < _eNat _mComma _eInt > _vPriorityClassMap =
		_eMap < _eNat _mComma _eInt >
		( _eSeq < _eInt >()._lAppend
		 	( 1, 2, 3, 4 )._lAppend( 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20 ),
		  _eSeq < _eInt >()._lAppend
		  	( IDLE_PRIORITY_CLASS, IDLE_PRIORITY_CLASS, IDLE_PRIORITY_CLASS, IDLE_PRIORITY_CLASS)._lAppend(
		  	  NORMAL_PRIORITY_CLASS, NORMAL_PRIORITY_CLASS, NORMAL_PRIORITY_CLASS, NORMAL_PRIORITY_CLASS,
			  HIGH_PRIORITY_CLASS, HIGH_PRIORITY_CLASS, HIGH_PRIORITY_CLASS, HIGH_PRIORITY_CLASS,
			  HIGH_PRIORITY_CLASS, REALTIME_PRIORITY_CLASS, REALTIME_PRIORITY_CLASS, REALTIME_PRIORITY_CLASS,
			  REALTIME_PRIORITY_CLASS, REALTIME_PRIORITY_CLASS, REALTIME_PRIORITY_CLASS, REALTIME_PRIORITY_CLASS
		  	)
		);

	static const _eMap < _eNat _mComma _eInt > _vPriorityLevelMap =
		_eMap < _eNat _mComma _eInt >
		( _eSeq < _eInt >()._lAppend
		  	( 1, 2, 3, 4 )._lAppend( 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20 ),
		  _eSeq < _eInt >()._lAppend
		  	( THREAD_PRIORITY_IDLE, THREAD_PRIORITY_LOWEST, THREAD_PRIORITY_BELOW_NORMAL, THREAD_PRIORITY_NORMAL)._lAppend(
		  	  THREAD_PRIORITY_LOWEST, THREAD_PRIORITY_BELOW_NORMAL, THREAD_PRIORITY_NORMAL, THREAD_PRIORITY_ABOVE_NORMAL,
			  THREAD_PRIORITY_LOWEST, THREAD_PRIORITY_BELOW_NORMAL, THREAD_PRIORITY_NORMAL, THREAD_PRIORITY_ABOVE_NORMAL,
			  THREAD_PRIORITY_HIGHEST, THREAD_PRIORITY_IDLE, THREAD_PRIORITY_LOWEST, THREAD_PRIORITY_BELOW_NORMAL,
			  THREAD_PRIORITY_NORMAL, THREAD_PRIORITY_ABOVE_NORMAL, THREAD_PRIORITY_HIGHEST, THREAD_PRIORITY_TIME_CRITICAL
		  	)
		);

	bool setCurrentThreadPriority(const _eNat priority)
	{
		// Can we put a pre-condition in her that priority <= 20 ???
		// Set current process base priority so that task manager can see the changed priority...
		HANDLE hCurProcHndl = GetCurrentProcess();
		BOOL bClassSuccess = SetPriorityClass(hCurProcHndl, static_cast<DWORD>(_vPriorityClassMap[priority]));
		// Set current thread priority to actually do the job!
		HANDLE hCurrentThread = GetCurrentThread();
		BOOL bThreadSuccess = SetThreadPriority(hCurrentThread, static_cast<DWORD>(_vPriorityLevelMap[priority]));

		return bClassSuccess && bThreadSuccess;
	}

	//------------------------------------------------------------------------------
	// Function: _eosInterface :: getLocalTime()
	// Purpose:  Get the current local date and time, based on the current time-zone
	//			 settings.
	//------------------------------------------------------------------------------
	Time getLocalTime()
	{
		SYSTEMTIME lt;
		::GetLocalTime(&lt);
		return Time( lt.wSecond, lt.wMinute, lt.wHour, lt.wDayOfWeek, lt.wDay, lt.wMonth, lt.wYear );
	}

	//------------------------------------------------------------------------------
	// Function: _eosInterface :: getImageVersion(const _eNativeChar *moduleName = 0)
	// Purpose:  Get the version information for an executable image (this
	//           information will have been stored in the 'ProductVersion' field of
	//           the version resource of the module in question) as a sequence of
	//			 integers (4 integers for Win32). If no argument is passed, then the
	//			 module queried is the currently running executable, else a string
	//			 can be passed that is the name of another currently loaded module
	//			 (loaded into the currently running executable's address space, that
	//			 is), for example the name of a DLL such as "msvcrt.dll" ...
	//------------------------------------------------------------------------------
	_eSeq < _eInt > getImageVersion(const _eNativeChar *moduleName)
	{
		_eNativeChar thePath[MAX_PATH+1];
		::GetModuleFileName(::GetModuleHandle(moduleName), thePath, MAX_PATH+1);
		DWORD theHandle = 0;
		DWORD theSize = ::GetFileVersionInfoSize(thePath, &theHandle);
		WORD theV1 = 0, theV2 = 0, theV3 = 0, theV4 = 0;
		if (theSize != 0)
		{
			VS_VERSIONINFO* theVI = (VS_VERSIONINFO*)::malloc(theSize);
			VS_FIXEDFILEINFO* theFFI = 0;
			UINT theLen = 0;
			if (   ::GetFileVersionInfo(thePath, theHandle, theSize, theVI)
				&& ::VerQueryValue(theVI, _mCstr("\\"), (LPVOID*)&theFFI, &theLen)
				&& theLen != 0)
			{
				theV1 = HIWORD(theFFI->dwFileVersionMS);//dwProductVersionMS);
				theV2 = LOWORD(theFFI->dwFileVersionMS);//dwProductVersionMS);
				theV3 = HIWORD(theFFI->dwFileVersionLS);//dwProductVersionLS);
				theV4 = LOWORD(theFFI->dwFileVersionLS);//dwProductVersionLS);
			}
			free(theVI);
		}
		return _eSeq < _eInt > ()._lAppend (theV1, theV2, theV3, theV4);
	}

	//---------------------------------------------------------------------------------------------
	// Function: _eosInterface :: getOsInfo()
	// Purpose:  Determine which operating system is currently running. Return details of OS-type
	//           (windows95, 98, NT etc) and major and minor OS-version numbers
	//---------------------------------------------------------------------------------------------
	_eHndl<OsInfo> getOsInfo()
	{
		// We already got the operating system version info during initialisation
	   	if (osVersionInfoType == 0)
		{
	      	return _eHndl<OsInfo>(new OsInfo(OsType::unknown, 0, 0));
		}
		else
		{
			switch (osvi.dwPlatformId)
			{
	   			case VER_PLATFORM_WIN32_NT:
				{
					if (osVersionInfoType == 2)	// if service pack information available ...
						return _eHndl<OsInfo>(new OsInfo(OsType::windowsNT, osvi.dwMajorVersion, osvi.dwMinorVersion, osvi.wServicePackMajor, osvi.wServicePackMinor));
					else
						return _eHndl<OsInfo>(new OsInfo(OsType::windowsNT, osvi.dwMajorVersion, osvi.dwMinorVersion));
				}
	   			case VER_PLATFORM_WIN32_WINDOWS:
				{
	    	  		if ((osvi.dwMajorVersion > 4) || ((osvi.dwMajorVersion == 4) && (osvi.dwMinorVersion > 0)))
						return _eHndl<OsInfo>(new OsInfo(OsType::windows98, osvi.dwMajorVersion, osvi.dwMinorVersion));
	    	  		else
						return _eHndl<OsInfo>(new OsInfo(OsType::windows95, osvi.dwMajorVersion, osvi.dwMinorVersion));
				}
	   			default:
					return _eHndl<OsInfo>(new OsInfo(OsType::unknown, 0, 0));
			}
		}
	}

	//---------------------------------------------------------------------------------------------
	// Function: genNormalizedFilePathObject(const _eNativeChar *path, const _eNativeChar *fname)
	// Purpose:  Given a pathname and filename, build an FilePath object that contains them
	//           in an expanded form if this is possible (by 'expanded' form we mean the 'long'
	//           filename format as opposed to the short form containing tildas). Note that the
	//           expanded form is only available for the path or file if the path or file exists
	//           and we are running under Windows2000; if the expanded form of either the pathname
	//           or filename does NOT exist, we leave it unchanged.
	//---------------------------------------------------------------------------------------------
	_eHndlUnionVoid<FilePath> genNormalizedFilePathObject(const _rstring path, const _rstring fname)
	{
		DWORD osCallResult;
		_eCstring full(path._oPlusPlus(fname));		// glue the path and file together ready to expand any non-longfilename segments
		_eNativeChar *shortPath = full.changeStr();
		_eNativeChar longPath[MAX_PATH];
		osCallResult = getLongPathName(shortPath, longPath, MAX_PATH);	// note that this could just be 'GetLongPathName' under Windows2000
		if (osCallResult == 0 || osCallResult >= MAX_PATH)
		{	// Failed to expand the path - signal this as an error ...
			return _eHndlUnionVoid<FilePath>();
		}
		// OK, we've sucessfully expanded the full pathname; now we need to re-seperate the
		// path and filename components ...
		_eNativeChar *fileptr = NULL;
		_eNativeChar buffer[MAX_PATH];
		osCallResult = GetFullPathName(longPath, MAX_PATH, buffer, &fileptr);
		if (osCallResult == 0 || osCallResult >= MAX_PATH)
		{	// Failed to re-seperate the path and filename components - signal this as an error ...
			return _eHndlUnionVoid<FilePath>();
		}
		// OK, all done, so just need to populate and return the 'FilePath' object ...
		if (fileptr == NULL)
		{
			return _eHndlUnionVoid<FilePath>(new FilePath(_lString(buffer), _mString("")));
		}
		else
		{
			const _rstring file = _lString(fileptr);
			*fileptr = '\0';
			return _eHndlUnionVoid<FilePath>(new FilePath(_lString(buffer), file));
		}
	}

	// Since 'GetLongPathName(..)' cannot even appear in an executable to be run under anything but Windows98
	// or 2000 (we get an error from the loader which I had previously thought was a runtime error from our
	// executable), we have to provide our own version of it here.
	// Notes:
	// (1) If expansion fails at any particular path segment, we return the partially expanded path, ie. expanded upto the
	//     point of failure, and then as was afterwards (I'm not sure what the MS version would do in this situation);
	// (2) The function will not handle UNC paths as at 13/12/2000.
	//
	// The GetLongPathName function converts the specified path to its long form. If no long path
	// is found, this function simply returns the specified name.
	// Return Values:
	// - If the function succeeds, the return value is the length of the string copied to the longPath parameter,
	//   in characters. This length does not include the terminating null character;
	// - If longPath is too small, the function returns the size of the buffer required to hold the long path, in characters.

	DWORD getLongPathName(const _eNativeChar *shortPath, _eNativeChar *longPath, DWORD maxBufLen)
	{
		_eNativeChar *termSlash, *prevSlash, *lShortPathPtr;
		HANDLE hFindFile;
		WIN32_FIND_DATA FindFileData;
		// ensure the path has at least the drive-identifier (eg. "C:\")
		if (_mStrlen(shortPath) < 3 || maxBufLen < 3)
		{
			_mStrcpy(longPath, shortPath);
			return 0;										// this is an error condition
		}
		// allocate and initialize a buffer to hold the shortpath plus a trailing slash and null terminator ...
		_eCstring lShortPath(_mStrlen(shortPath) + 2);
		lShortPathPtr = lShortPath.changeStr();
		_mStrcpy(lShortPathPtr, shortPath);
		_mStrcat(lShortPathPtr, _mCstr("\\"));
		// allocate and prepare a buffer to hold the expanded path ...
		_rstring lLongPath;
		lLongPath = lLongPath.append(lShortPathPtr[0]);
		lLongPath = lLongPath.append(lShortPathPtr[1]);					// the expanded path will begin with the drive identifier
		// parse the modified shortpath passed, segment by segment, expanding as we go ...
		termSlash = _mStrchr(lShortPathPtr + 3, (int)'\\');				// find the first slash after the drive identifier and root slash
		prevSlash = lShortPathPtr + 2;
		while (termSlash != (_eNativeChar *)NULL)
		{
			*termSlash = '\0';											// terminate the path here, ready to expand this segment
			// expand the last-segment in the current path chunk ...
	//		printf("\nExpanding '%s' ...", lShortPathPtr);
			hFindFile = FindFirstFile(lShortPathPtr, &FindFileData);
			*termSlash = '\\';											// restore the path
			if (hFindFile == INVALID_HANDLE_VALUE)
			{
				// failed, so we just append the rest of the original shortpath and return ...
				*(lShortPathPtr + _mStrlen(shortPath)) = '\0';		// re-terminate the copy of the shortpath so that we remove the terminating slash
	//			printf("\nInfo: terminating after failed find call (remaining path was '%s').", prevSlash);
				lLongPath = lLongPath._oPlusPlus(_lString(prevSlash));
				FindClose(hFindFile);
				break;
			}
			else
			{
				lLongPath = lLongPath.append('\\');
				lLongPath = lLongPath._oPlusPlus(_lString(FindFileData.cFileName));
			}
			FindClose(hFindFile);
			// find the slash the terminates the next path segment ...
			prevSlash = termSlash;
			termSlash = _mStrchr(termSlash + 1, (int)'\\');
		}
	//	printf("\nInfo: length of expanded path = %d; max buffer size = %d", lLongPath._oHash(), maxBufLen);
		if (static_cast<DWORD>(lLongPath._oHash()) > maxBufLen)
		{
			return static_cast<DWORD>(lLongPath._oHash());				// error! target buffer supplied not large enough
		}
		_mStrcpy(longPath, _eCstring(lLongPath).str());
	//	printf("\ngetLongPathName returned: '%s'\n", longPath);
		return static_cast<DWORD>(_mStrlen(longPath));
	}

}			// end namespace _eosInterface

//----------------------------------------------------------------------------
// Function: clock()
// Purpose:  Obtain the amount of processor time used by the calling process.
//           The 'clocksPerSecond' function will return the units used.
//----------------------------------------------------------------------------
_eNat Environment :: clock() const
{
	if (_eosInterface :: canGetProcessTimes)
	{
		// Because clock() under Windows returns wall-clock time, we prefer to call GetProcessTimes()
		FILETIME creationTime, exitTime, kernelTime, userTime;
		if (GetProcessTimes(_eosInterface :: currentProcess, &creationTime, &exitTime, &kernelTime, &userTime))
		{
			_ULARGE_INTEGER temp;
			temp.u.LowPart = userTime.dwLowDateTime;
			temp.u.HighPart = userTime.dwHighDateTime;
			return static_cast<_eNat>(temp.QuadPart / 10000);	// divide by 10,000 to convert 100ns to 1ms increments
		}
		else
		{
			_eosInterface :: canGetProcessTimes = false;
		}
	}
	return static_cast<_eNat>(:: clock());		// this is wall-clock time under Windows!
}

//-----------------------------------------------------------------------------------------
// Function: Environment :: clocksPerSecond() const
// Purpose:  Obtain the units (as a fraction of a second) returned by the 'clock' function.
//-----------------------------------------------------------------------------------------
_eNat Environment :: clocksPerSecond() const
{
	if (_eosInterface :: canGetProcessTimes)
	{
		return 1000;								// we count in milliseconds, so 2^31 of them represents 24 days
	}
	else
	{
		return static_cast<_eNat>(CLOCKS_PER_SEC);
	}
}

// End.
