// **************************************************************************
// * File:   CommandExecuter.cpp											*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2002 Escher Technologies Ltd.								*
// * Desc:   Code to run external commands from Perfect						*
// **************************************************************************

#include "Ertsys.hpp"
#include "CharHelpers.hpp"
#include <cwchar>
#include <cstdlib>			// for wcstombs

#if defined(_MSC_VER)
	#include <io.h>			// for _dup and _dup2
	#include <process.h>	// for spawn
	#define DUP _dup
	#define DUP2 _dup2
	#define FILENO _fileno
#else
	#include <unistd.h>		// for dup, dup2 and fork
	#include <fcntl.h>		// for fcntl
	#include <sys/wait.h>	// for wait
	#define DUP dup
	#define DUP2 dup2
	#define FILENO fileno
#endif

#include <cerrno>			// for error number constants

// Changes the file associated with 'toRedirect' to be 'newFile', returning a handle to the original
// file, or -1 if the operation fails.
int redirectStream(const int toRedirect, const FileRef * newFile)
{
	// Attach the handle 'duplicate' to the same file as 'toRedirect'
	int duplicate = DUP( toRedirect );
   	if( duplicate == -1 )
   	{
		// Error duplicating handle (no more handles availiable)
		return duplicate;
	}
	// Attach 'toRedirect' to the given file
	int temp = DUP2( FILENO(newFile->getFileHandle()), toRedirect );
	if( temp == -1 )
	{
		// Error reassigning handle...
		return -1;
	}
	return duplicate;
}

void Environment :: execute (const _rstring command, const _eSeq < _rstring > args, const
    FileRef * stdinfile, const FileRef * stdoutfile, const FileRef * stderrfile, FileError :: _eEnum & err)
{
    _mSchema (execute);
	// Flush everything so that we don't get any strange results (this is required before doing a 'spawn' anyway)...
	fflush(NULL);
	// We may need to redirect stdin, stdout and stderr...
	int originalStdin;
	int originalStdout;
	int originalStderr;
	const int stdinFileHandle = FILENO(stdin);
	const int stdoutFileHandle = FILENO(stdout);
	const int stderrFileHandle = FILENO(stderr);

	if (stdinfile != NULL)
	{   originalStdin = redirectStream(stdinFileHandle, stdinfile);
		if (originalStdin == -1)
		{	err = FileError :: otherError;
			return;
		}
	}
	if (stdoutfile != NULL)
	{   originalStdout = redirectStream(stdoutFileHandle, stdoutfile);
		if (originalStdout == -1)
		{	err = FileError :: otherError;
			return;
		}
	}
	if (stderrfile != NULL)
	{   originalStderr = redirectStream(stderrFileHandle, stderrfile);
		if (originalStderr == -1)
		{	err = FileError :: otherError;
			return;
		}
	}

	// Generate a c-string array for the arguments to the command
	_eCstring cmd(command);
	const _eInt numArgs = args._oHash();

#if defined(_MSC_VER)
	// Need two extra 'arguments' - arg[0] should be the command name, and arg[n+1] should be NULL
	_eNativeChar** argPtrs = new _eNativeChar*[numArgs+2];
	argPtrs[0] = cmd.changeStr();
	argPtrs[numArgs+1] = NULL;

	// Now copy all the real arguments into the array
	for (int i = 0; i < numArgs; i++)
	{
		_rstring nextArg = args[i];
		const _eInt numChars = nextArg._oHash();
		argPtrs[i+1] = new _eNativeChar[numChars+1];
		for (int j = 0; j < numChars; j++)
		{
			argPtrs[i+1][j] = nextArg[j];
		}
		argPtrs[i+1][numChars] = '\0';
	}

	// Spawn the actual process and wait for it to finish
  #if _MSC_VER >= 1400
	const intptr_t spawnRes = _mSpawnvp(_P_WAIT, cmd.str(), argPtrs);			// no error check here
  #else
	const int spawnRes = _mSpawnvp(_P_WAIT, cmd.str(), argPtrs);				// no error check here
  #endif

	if (spawnRes == -1)
	{	switch(errno)
		{	case ENOENT:
				err = FileError :: fileNotFound;
				break;
			case ENOEXEC:
				err = FileError :: attribError;		// not executable
				break;
			default:
				err = FileError :: otherError;
		}
	}
	else
	{
		err = FileError :: success;
	}

	// Delete the memory we allocated for the arguments
	for (int i = 1; i <= numArgs; i++)
	{
		delete[] argPtrs[i];
	}
	delete[] argPtrs;

#else

	// Linux doesn't have a wide version of execvp so we have to construct narrow arguments
	// Need two extra 'arguments' - arg[0] should be the command name, and arg[n+1] should be NULL
	char** argPtrs = new char*[numArgs+2];

#if defined(_dUnicode)
	size_t len = wcstombs(NULL, cmd.str(), 0) + 1;
	argPtrs[0] = new char[len];
	wcstombs(argPtrs[0], cmd.str(), len);

	// Now copy all the real arguments into the array
	for (int i = 0; i < numArgs; i++)
	{
		_eCstring nextArg(args[i]);
		size_t len = wcstombs(NULL, nextArg.str(), 0) + 1;
		argPtrs[i] = new char[len];
		wcstombs(argPtrs[i], nextArg.str(), len);
	}
#else
	size_t len = strlen(cmd.str()) + 1;
	argPtrs[0] = new char[len];
	strncpy(argPtrs[0], cmd.str(), len);

	// Now copy all the real arguments into the array
	for (int i = 0; i < numArgs; i++)
	{
		_rstring nextArg = args[i];
		const _eInt numChars = nextArg._oHash();
		argPtrs[i+1] = new char[numChars+1];
		for (int j = 0; j < numChars; j++)
		{
			argPtrs[i+1][j] = nextArg[j];
		}
		argPtrs[i+1][numChars] = '\0';
	}
#endif
	argPtrs[numArgs+1] = NULL;

	// Create a pipe to pass back any error from exec
	int fildes[2];
	if (pipe(fildes) == -1)
	{	// failed to create pipe
		err = FileError :: otherError;
		return;
	}
	// Fork execution
	pid_t childProcess = fork();
	if (childProcess == -1)
	{	// failed to fork
		err = FileError :: otherError;
		::close(fildes[0]);						// note global close hidden by 'close' methods in Environment
		::close(fildes[1]);
	}
	else if (childProcess == 0)
	{	// we are in the child fork, so we should try to execute the desired command
		::close(fildes[0]);						// don't need the 'read' part of the pipe here
		fcntl(fildes[1], F_SETFD, FD_CLOEXEC);  // set the 'write' part to close upon 'exec'
#if defined(_dUnicode)
		char* buf;
		wcs2ncs(buf, cmd.str());
		int execRes = execvp(buf, argPtrs);
#else
		int execRes = execvp(cmd.str(), argPtrs);
#endif
		// if we got here, the exec failed, so we should send the appropriate error down the pipe
		switch(errno)
		{	case ENOENT:
			case ENOTDIR:
				err = FileError :: fileNotFound;
				break;
			case EACCES:
				err = FileError :: attribError;		// not executable, or can't search requested directory
				break;
			default:
				err = FileError :: otherError;
		}
		// write the error to the pipe - no error check here, because if it fails there's nothing
		// much we can do anyway since this is the only way of getting error messages back to the parent
		(void)::write(fildes[1], &err, sizeof(FileError::_eEnum));	// write is hidden by Environment::write
		::close(fildes[1]);
		_exit(execRes);
	}
	else
	{	// we are the parent fork, so we should wait for the child fork to finish
		::close(fildes[1]);	// don't need the 'write' part of the pipe here
		int waitRes;
		waitpid(childProcess, &waitRes, 0);
		// now see if there was anything in the pipe - if so we had an error
		// read is hidden by method in Environment, so need global scope resolution
		if (::read(fildes[0], &err, sizeof(FileError::_eEnum)) == 0)
		{
			// if not then the child process must have been successful
			err = FileError :: success;
		}
		::close(fildes[0]);
	}

	// Delete the memory we allocated for the arguments
	for (int i = 0; i <= numArgs; i++)
	{
		delete[] argPtrs[i];
	}
	delete[] argPtrs;
#endif

	// Put the standard i/o streams back the way they were...
	if (stdinfile != NULL)
	{
		// Flush the stream and then put back the original file to the stdOutFileHandle...
		fflush( stdin );
		DUP2( originalStdin, stdinFileHandle );	// no error check here
	}
	if (stdoutfile != NULL)
	{
		// Flush the stream and then put back the original file to the stdOutFileHandle...
		fflush( stdout );
		DUP2( originalStdout, stdoutFileHandle );	// no error check here
	}
	if (stderrfile != NULL)
	{
		// Flush the stream and then put back the original file to the stdOutFileHandle...
		fflush( stderr );
		DUP2( originalStderr, stderrFileHandle );	// no error check here
	}
}

// End.
