// **************************************************************************
// * File:   LinuxSystemCalls.cpp											*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Code to implement an interface between runtime and the			*
// *         target OS.														*
// **************************************************************************

//---------
// Includes
//---------

#include "Ertsys.hpp"

#include <cwchar>
#include <sys/stat.h>
#include <ctime>
#include <utime.h>
#include <sys/resource.h>
#include <cerrno>
#include <stdlib.h>

#include "LinuxSystemCalls.hpp"
#include "CharHelpers.hpp"

namespace _eosInterface
{
	int ResolvePath(_eNativeChar *);
	int GetFullPathName(const _eNativeChar *pFileName, int nBufLen, _eNativeChar *pBuffer, _eNativeChar **pFilePart);
}

//---------------------------------------------------------------------------------
// Function: _eosInterface :: Initialize()
// Purpose: Set up any static data that is needed.
//---------------------------------------------------------------------------------
void _eosInterface :: initialize()
{
	// No initialisation needed yet for Linux
}

//---------------------------------------------------------------------------------
// Function: _eosInterface :: normalizefilename(const char *path, const char *fname)
// Purpose:  Given a filename and a default path ("path"), split it into a full
//           path and a filename ("fname") without path. The default path must
//           either be empty (in which case the current directory is assumed)
//           or end in "\". The split parts are held in the FilePath object
//           returned on success, esle null is returned.
//---------------------------------------------------------------------------------
_eHndlUnionVoid<FilePath> _eosInterface :: normalizefilename(const _eNativeChar *path, const _eNativeChar *fname)
{
	_eNativeChar buffer[NAME_MAX];
	_eNativeChar *fileptr = NULL;
	int getFullPathNameRes;

	if (fname[0] == _mCstr('/'))
	{
		// The filename includes a path from the root,
		// so just normalise the filename...
		getFullPathNameRes = _eosInterface :: GetFullPathName(fname, NAME_MAX, buffer, &fileptr);
	}
	else
	{
		// No root path name given so use the default path
		_eCstring tempresult(_mStrlen(path) + _mStrlen(fname) + 1);  // temp buffer
		_mStrcpy(tempresult.changeStr(), path);
		_mStrcat(tempresult.changeStr(), fname);
		getFullPathNameRes = _eosInterface :: GetFullPathName(tempresult.str(), NAME_MAX, buffer, &fileptr);
	}
	// Check if GetFullPathName call succeeded ...
	if (getFullPathNameRes == 0)				// if 'GetFullPathName(..)' failed ..
	{
		return _eHndlUnionVoid<FilePath>();		// returning null-handle for failure
	}
	// "buffer" now contains the full path and filename, and "fileptr" points
	// to the final file component in it
	if (fileptr == NULL)
	{
		return _eHndlUnionVoid<FilePath>(new FilePath(_lString(buffer), _mString("")));
	}
	else
	{
		const _rstring file = _lString(fileptr);
		*fileptr = _mCstr('\0');
		return _eHndlUnionVoid<FilePath>(new FilePath(_lString(buffer), file));
	}
}

//---------------------------------------------------------------------------------
// Make chmod call to set file attributes
//---------------------------------------------------------------------------------
FileError :: _eEnum _eosInterface :: chmod(const _eNativeChar *fname, _eSet<FileAttribute :: _eEnum> atts)
{
	mode_t mode = atts._ovIn(FileAttribute :: read) ? S_IRUSR | S_IRGRP | S_IROTH : 0;
	if (atts._ovIn(FileAttribute :: write))
		mode = mode | S_IWUSR | S_IWGRP | S_IWOTH;
	if (atts._ovIn(FileAttribute :: execute))
		mode = mode | S_IXUSR | S_IXGRP | S_IXOTH;
	int temp = _mChmod(fname, mode);
	return    temp == 0			? FileError :: success
			: errno == ENOENT 	? FileError :: fileNotFound
			: errno == EPERM 	? FileError :: permError
			: FileError :: otherError;
}

//-------------------------------------------------------------------
// Function: _eosInterface :: caseSensitiveFileNames()
// Purpose:  Obtain information from the OS as to whether or not the
//           local filessystem is case sensitive.
//-------------------------------------------------------------------
bool _eosInterface :: caseSensitiveFileNames()
{
	return true;				                        // file open under UNIX is case sensitive
}

//-------------------------------------------------------------------
// Function: _eosInterface :: getPathSeparator()
// Purpose:  Obtain information from the OS as to what the path
//           separator character is on the local filessystem.
//-------------------------------------------------------------------
_eNativeChar _eosInterface :: getPathSeparator()
{
	return _mCstr('/');
}

//----------------------------------------------------------------------------------------------------
// Function: _eosInterface :: setCurrentThreadPriority(const _eNat priority)
// Purpose:  Set the priority of the current process to that specified. 'Perfect' supports 20 priority
//           levels. For more information, see manual entry for 'setPriority'.
//----------------------------------------------------------------------------------------------------

bool _eosInterface :: setCurrentThreadPriority(const _eNat priority)
{
	return (setpriority(PRIO_PROCESS, 0, 0) == 0);
}

//------------------------------------------------------------------------------
// Function: _eosInterface :: getLocalTime()
// Purpose:  Get the current local date and time, based on the current time-zone
//			 settings.
//------------------------------------------------------------------------------
Time _eosInterface :: getLocalTime()
{
	time_t secsSinceEpoch;
	struct tm* timeStruct;

	time(&secsSinceEpoch);		// note that this could error (returning -1 if so, but POSIX(1) doesn't specify any errors)
	timeStruct = localtime(&secsSinceEpoch);
	return Time( timeStruct->tm_sec, timeStruct->tm_min, timeStruct->tm_hour,
			   timeStruct->tm_wday, timeStruct->tm_mday, timeStruct->tm_mon, timeStruct->tm_year );
}

//------------------------------------------------------------------------------
// Function: _eosInterface :: getImageVersion(const char *moduleName)
// Purpose:  Get the version information for the running executable image
//------------------------------------------------------------------------------
_eSeq < _eInt > _eosInterface :: getImageVersion(const _eNativeChar *moduleName)
{
	//??? Don't know how we're going to do this yet...
	return _eSeq < _eInt > ()._lAppend (0, 0, 0, 0);
}

//---------------------------------------------------------------------------------------------
// Function: _eosInterface :: getOsInfo()
// Purpose:  Determine which operating system is currently running. Return details of OS-type
//           (Linux etc) and major and minor OS-version numbers ...
//---------------------------------------------------------------------------------------------
_eHndl<OsInfo> _eosInterface :: getOsInfo()
{
	return _eHndl<OsInfo>(new OsInfo(OsType::_rlinux, 0, 0));		//??? need to find out how to get the OS version
}

//********************************************
//* Extra routines required by the above ... *
//********************************************

#include <cstdio>		// for size_t
#include <unistd.h>
#include <cstring>

// ResolvePath, by Dommo, February 2000.
// Remove any path segment that comes before a '../' sequence
// Modify passed string, and returns one of the following integer values:
// 0 for success,
// 1 if had non-slash after '..' when not at end of path spec,
//// 2 if we had '../' when already at root directory.

int _eosInterface :: ResolvePath(_eNativeChar *spec)
{
	size_t spec_pos = 0;
	int path_pos = 0;
	int prev_slash_pos = 0;						// need this in case see '..\' or end of string

	//*******************************************************************
	// need to run through specified path, converting '.\' and '..\' etc
	//*******************************************************************

	while (spec_pos < _mStrlen(spec))
	{
		//printf("\nSpec_pos='%d' Path_pos='%d' Prev_slash_pos='%d'",spec_pos,path_pos,prev_slash_pos);

		if (spec[spec_pos] == _mCstr('.'))		// dot sequences maybe special ...
		{
			if (spec[spec_pos + 1] == _mCstr('.'))			// two dots together means go back a directory
			{
				if (spec[spec_pos + 2] != _mCstr('/'))		// missing slash after '..' ??
				{
					if (spec[spec_pos + 2] == _mCstr('\0'))
					{
						spec_pos -= 1;			// yes, but if at end of string, ok, so ensure 'spec_pos+=3' works ok
					}
					else
					{
						return 1;				// yes non-slash after '..' and not at end of string, so its nonsense !!
					}
				}

//				if(prev_slash_pos==0)
//					return 2;					// had '..\' when already at root !!

				path_pos = prev_slash_pos;		// had '..\' so move current end of path back to last '\'
				while( (prev_slash_pos > 0) && (spec[prev_slash_pos - 1] != _mCstr('/')) )
				{
					--prev_slash_pos;	// find pos'n of previous backslash
				}
				path_pos += 1;					// keep the '\' at the end of the new path
				spec_pos += 3; 					// actioned '..\' now, so move past it
			}
			else								// had dot followed by something else ...
			{
				if (spec[spec_pos + 1] == _mCstr('/'))
				{
					if ( (spec_pos > 0) && (spec[spec_pos - 1] != _mCstr(':')) && (spec[spec_pos - 1] != _mCstr('/')) )
						return 3;				// had a '.\' with no ':' or '\' preceding it, must be crap !!

					spec_pos+=2; 				// had '.\' so ignore !!
				}
				else
					spec[path_pos++] = spec[spec_pos++];	// a dot with no slashes or other dots next to it
			}
		}
		else
		{

			//*****************************************
			// ok, can copy this character to path ...
			//*****************************************

			if ((spec[path_pos++] = spec[spec_pos++]) == _mCstr('/'))	// copied a '\' ?
			{
				prev_slash_pos = path_pos-1;
				while ( (prev_slash_pos > 0) && (spec[prev_slash_pos - 1] != _mCstr('/')) )
				{
					--prev_slash_pos;					// find pos'n of previous backslash
				}
			}
		}
	}
	spec[path_pos] = _mCstr('\0');						// terminate the entire path string
	return 0;
}

// Unix version of Win32 function 'GetFullPathName(..)', by Dommo, February 2000
// Given a pathname (i.e. path and file spec), find the fully qualified pathname and the filename, based
// upon the pathname supplied (i.e. if it doesn't start with a '/', base result upon current directory)
// Modify buffer passed, and return one of the following integer values:
// 0 if had an error,
// size of fully qualified pathname returned in buffer, if successful.

int _eosInterface :: GetFullPathName(const _eNativeChar *pFileName, int nBufLen, _eNativeChar *pBuffer, _eNativeChar **pFilePart)
{
	// Check that buffer will at least fit the raw pathname passed-in!!
	if (static_cast<int>(_mStrlen(pFileName)) >= nBufLen)
	{
		return 0;		// non-starter!!
	}

	// OK, first we need to check whether the filename passed already specifies a volume If so,
	// we just need to find the position of the file part. If not, we need to get the current
	// directory and base the resultant pathname on that ...
	if (pFileName[0] == '/')
	{
		_mStrcpy(pBuffer, pFileName);
	}
	else
	{
		if (_mGetcwd(pBuffer, nBufLen) == NULL)
		{
			return 0;			// failed to get cwd (either buffer was not large enough or caller had no access to current path)
		}
		size_t dirLen = _mStrlen(pBuffer);
		if (dirLen == 1)
		{
			*pBuffer = _mCstr('\0');	// if at root directory, we need to avoid the '//' situation
		}
		if (static_cast<int>(dirLen + 1 + _mStrlen(pFileName)) >= nBufLen)
		{
			return 0;			// buffer was not large enough for <current directory> + '/' + <filename>
		}
		_mStrcat(pBuffer, _mCstr("/"));
		_mStrcat(pBuffer, pFileName);
	}
	// Remove ./ sequences and resolve <dir>/../ sequences
	if (_eosInterface :: ResolvePath(pBuffer) == 1)
	{
		return 0;				// Illegal '..' sequence found in pathname
	}
	// OK, now we have our fully qualified pathname. Now we need to find the start of the file part of the pathname
	for (size_t i = _mStrlen(pBuffer); i != 0; )
	{
		--i;
		if (*(pBuffer + i) == _mCstr('/'))
		{
			*pFilePart = pBuffer + i + 1;
			break;
		}
	}
	return _mStrlen(pBuffer);		// success!!
}

//----------------------------------------------------------------------------
// Function: clock()
// Purpose:  Obtain the amount of processor time used by the calling process.
//           The 'clocksPerSecond' function will return the units used.
//----------------------------------------------------------------------------
_eNat Environment :: clock() const
{
	return static_cast<_eNat>(:: clock());
}

//-----------------------------------------------------------------------------------------
// Function: Environment :: clocksPerSecond() const
// Purpose:  Obtain the units (as a fraction of a second) returned by the 'clock' function.
//-----------------------------------------------------------------------------------------
_eNat Environment :: clocksPerSecond() const
{
	return static_cast<_eNat>(CLOCKS_PER_SEC);
}

//-----------------------------------------------------------------------------------------------

#if defined(_dUnicode)
// glibc doesn't define wide character file system interface routines, so we define them here

#include <cstdio>

FILE *_wfopen(const wchar_t *filename, const wchar_t *mode)
{
	char* nfname;
	char* nmode;
	wcs2ncs(nfname, filename);
	wcs2ncs(nmode, mode);
	return fopen(nfname, nmode);
}

int _wrename(const wchar_t *oldname, const wchar_t *newname)
{
	char* noldname;
	char* nnewname;
	wcs2ncs(noldname, oldname);
	wcs2ncs(nnewname, newname);
	return rename(noldname, nnewname);
}

int _wunlink(const wchar_t *filename)
{
	char* nfname;
	wcs2ncs(nfname, filename);
	return unlink(nfname);
}

int _waccess(const wchar_t *path, int mode)
{
	char* npath;
	wcs2ncs(npath, path);
	return access(npath, mode);
}

int _wstat(const wchar_t *path, struct ::stat *buffer)
{
	char* npath;
	wcs2ncs(npath, path);
	return stat(npath, buffer);
}

int _wutime(const wchar_t *filename, struct ::utimbuf *times)
{
	char* nfname;
	wcs2ncs(nfname, filename);
	return utime(nfname, times);
}

wchar_t *_wgetcwd(wchar_t *buffer, int maxlen)
{
	char* nbuffer = new char[maxlen];		// assume no multibyte characters
	wchar_t* ret = NULL;
	const char* nret = getcwd(nbuffer, maxlen);
	if (nret != NULL)
	{
		int len = mbstowcs(buffer, nbuffer, maxlen);
		if (len > 0 && len < maxlen) 
		{
			ret = buffer;
		}
	}
	delete[] nbuffer;
	return ret;
}

int _wchdir(const wchar_t *dirname)
{
	char* ndirname;
	wcs2ncs(ndirname, dirname);
	return chdir(ndirname);
}

int _wchmod(const wchar_t *filename, int pmode)
{
	char* nfname;
	wcs2ncs(nfname, filename);
	return chmod(nfname, pmode);
}

int _wmkdir(const wchar_t *dirname, __mode_t m)
{
	char* ndirname;
	wcs2ncs(ndirname, dirname);
	return mkdir(ndirname, m);
}

#endif


// End.
