// **************************************************************************
// * File:   Swap.hpp														*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Global functions to swap values.								*
// **************************************************************************

#if !defined(_h_Swap)
#define _h_Swap

// We require a mechanism to swap two values of any type.
// We provide a general template version, and some specific versions for optimising common cases.

// Here is the general mechanism. In theory, we could use algorithm::swap from the STL,
// however if we try to do this with Microsoft VC++, we get into trouble because the include file
// <algorithm> refers to undefined symbols; so we define our own version instead.

// This generic version uses memcpy instead of assignment, to avoid manipulating use counts
template <class X>
void _lswap(X& x, X& y)
{
	char buffer[sizeof(X)];
	memcpy(buffer, &x, sizeof(X));		// copy x to buffer
	memcpy(&x, &y, sizeof(X));			// copy y to x
	memcpy(&y, buffer, sizeof(X));		// copy buffer to y
}

// Now for some specialisations. First a macro to make generating them easy.
#define _mDefineSwap(T)				\
	inline void _lswap(T& x, T& y)	\
	{                               \
		const T temp = x;           \
		x = y;                      \
		y = temp;                   \
	}

// Now generate them for common classes having inexpensive assignment operators
_mDefineSwap(_eInt)
_mDefineSwap(_eChar)
_mDefineSwap(_eByte)
_mDefineSwap(_eBool)
_mDefineSwap(_eReal)

// There is also a specialisation of 'swap' in BaseHandle.hpp

// We can use _mDefineSwap to define swap schemas for enumerations in generated code, if desired

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End
