// **************************************************************************
// * File:   ExceptionClasses.hpp											*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Definition of the debugging exception hierarchy (for pre-		*
// *         condition failure etc).									 	*
// **************************************************************************

#if !defined(_h_Cexcept)
#define _h_Cexcept

//-------------------------------------------------
// Class '_xDebug' - debugging exception base class
//-------------------------------------------------

class _xDebug : public _xCeption
{
  public:

  #if _dCallStack

	_eCallStackInfo CallStackInfo;

	void appendMessage(_eWorkspace ws, _eSize& size) const
	{
		append(ws, _mCstr("Perfect source code:"), size);
		CallStackInfo.appendSelf(ws, size);
	}

    _xDebug(const _eCallStackInfo& info, _eCstr file, _eNat line)
		: _xCeption(file, line), CallStackInfo(info)
	{
	}

  #else
	void appendMessage(_eWorkspace ws, _eSize& size) const
	{
	}

	_xDebug(_eCstr file, _eNat line)
		: _xCeption(file, line)
	{
	}

  #endif
};

//---------------------------------------------------------
// Class '_xPrecondition' - pre-condition failure exception
//---------------------------------------------------------

#if !defined(NDEBUG)

  class _xPrecondition : public _xDebug
  {
    protected:
	  void appendHeader(_eWorkspace buffer, _eSize& size) const
	  {
		  append(buffer, _mCstr("Precondition failure\n"), size);
	  }

    public:
	  _xPrecondition(const _eCallStackInfo& info, _eCstr file_name, _eNat line)
	  	  : _xDebug(info, file_name, line)
	  {
	  }
  };

  // Macros to check a precondition
  #define _mBeginPre \
	if (_eEscherRuntime::EnablePre && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting)	\
		_mBeginCheck

  #define _mCheckPre(condition, position) \
	  if (!(condition))	throw _xPrecondition(_eCallStackInfo(_tSelf.functionname, _mCstr(position)), _mCurrentFile , __LINE__);	\
	  _mEndCheck

  // Macros to check an enumeration constructor precondition
  #define _mBeginEnumPre \
	if (_eEscherRuntime::EnablePre && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting)	\
		_mBeginCheck

  #define _mCheckEnumPre(condition, position) \
	  if (!(condition))	throw _xPrecondition( \
			_eCallStackInfo(_eCstring(_mString("Enum CTOR (")._oPlusPlus(_lString(_tSelf.functionname)).append(_mCstr(')'))).str(), _mCstr(position)),	\
			_mCurrentFile , __LINE__);	\
	  _mEndCheck

#else

  #define _mBeginPre 							_mNoact
  #define _mCheckPre(condition, position)		_mNoact
  #define _mBeginEnumPre						_mNoact
  #define _mCheckEnumPre(condition, position)	_mNoact

#endif

//-----------------------------------------------------------
// Class '_xPostcondition' - post-condition failure exception
//-----------------------------------------------------------

#if !defined(NDEBUG)

  class _xPostcondition : public _xDebug
  {
    protected:
	  void appendHeader(_eWorkspace ws, _eSize& size) const
	  {
		  append(ws, _mCstr("Postcondition failed\n"), size);
	  }

    public:
	  _xPostcondition(const _eCallStackInfo& info, _eCstr file_name, _eNat line)
		  : _xDebug(info, file_name, line)
	  {
	  }
  };

  // Macros to check a postcondition
  #define _mBeginPost \
	if (_eEscherRuntime::EnablePost && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting)	\
		_mBeginCheck

  #define _mCheckPost(condition, position) \
	  if (!(condition))	throw _xPostcondition(_eCallStackInfo(_tSelf.functionname, _mCstr(position)), _mCurrentFile, __LINE__);	\
	  _mEndCheck

#else

  #define _mBeginPost						_mNoact
  #define _mCheckPost(condition, position)	_mNoact

#endif

//---------------------------------------------------------
// Class '_xConstraint' - type constraint failure exception
//---------------------------------------------------------

#if !defined(NDEBUG)

  class _xConstraint : public _xDebug
  {
    protected:
	  void appendHeader(_eWorkspace buffer, _eSize& size) const
	  {
		  append(buffer, _mCstr("Type constraint failure\n"), size);
	  }

    public:
	  _xConstraint(const _eCallStackInfo& info, _eCstr file_name, _eNat line)
	  	  : _xDebug(info, file_name, line)
	  {
	  }
  };

  // Macro to check a constraint
  #define _mBeginConstraint \
	if (_eEscherRuntime::EnableConstraints && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting)	\
		_mBeginCheck

  #define _mCheckConstraint(condition, position) \
	  if (!(condition))	throw _xConstraint(_eCallStackInfo(_tSelf.functionname, _mCstr(position)), _mCurrentFile , __LINE__);	\
	  _mEndCheck

#else

  #define _mBeginConstraint							_mNoact
  #define _mCheckConstraint(condition, position)	_mNoact

#endif

//------------------------------------------------------------------
// Class '_xThatOrAny' - 'that' or 'any' construct failure exception
//------------------------------------------------------------------

#if !defined(NDEBUG)

  class _xThatOrAny : public _xDebug
  {
    protected:
	  void appendHeader(_eWorkspace ws, _eSize& size) const
	  {
		  append(ws, _mCstr("Failed 'that' or 'any' expression\n"), size);
	  }

    public:
	  _xThatOrAny(const _eCallStackInfo& info, _eCstr file_name, _eNat line)
		  : _xDebug(info, file_name, line)
	  {
	  }
  };

  // Macros to check for a failed 'that' or 'any' expression
  // Note: this is enabled by the 'last choice' debug enable flag
  #define _mBeginThatOrAny \
	if (_eEscherRuntime::EnableLastChoice && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting)	\
		_mBeginCheck

  #define _mCheckThatOrAny(condition, position) \
	  if (!(condition))	throw _xThatOrAny(_eCallStackInfo(_tSelf.functionname, _mCstr(position)), _mCurrentFile, __LINE__);	\
	  _mEndCheck

#else

  #define _mBeginThatOrAny						_mNoact
  #define _mCheckThatOrAny(condition, position)	_mNoact

#endif

//-----------------------------------------------------------
// Class '_xLoopInvariant' - loop-invariant failure exception
//-----------------------------------------------------------

#if !defined(NDEBUG)

  class _xLoopInvariant : public _xDebug
  {
    protected:
	  void appendHeader(_eWorkspace ws, _eSize& size) const
	  {
		  append(ws, _mCstr("Loop Invariant failed\n"), size);
	  }

    public:
	  _xLoopInvariant(const _eCallStackInfo& info, _eCstr file_name, _eNat line)
		  : _xDebug(info, file_name, line)
	  {
	  }
  };

  // Exception class used to indicate that we have a recursive loop invariant check
  class _xRecursiveLoopInvariant
  {
    public:
	  _xRecursiveLoopInvariant() {}
  };

  // Macros to check a loop invariant.
  // Because a loop invariant may refer to the function etc. that contains the loop,
  // we must take care to avoid recursion. We do this by having a static recursion flag for
  // every loop invariant check, which we use to detect recursion. When we do detect recursion,
  // we throw an exception, which we catch locally to abort the loop invariant check and
  // carry on normally.
  // We have to explicitly save abd restore the check nesting count since if the recursive invariant
  // exception is thrown, the nesting count may not be correct when we catch the exception
  #define _mBeginLoopInvariant \
	if (_eEscherRuntime::EnableLoopInvariant && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting) \
	{ static bool _lNested = false; \
	  if (_lNested) throw _xRecursiveLoopInvariant(); \
	  _lNested = true; \
	  const unsigned int _lSaveNesting = _eEscherRuntime::CurrentCheckNesting++; \
	  try \
	  {

  #define _mCheckLoopInvariant(condition, position) \
		if (!(condition)) throw _xLoopInvariant(_eCallStackInfo(_tSelf.functionname, _mCstr(position)), _mCurrentFile, __LINE__); \
	  } catch (_xRecursiveLoopInvariant) {} catch (_xCannotEvaluate) {} \
	  _eEscherRuntime::CurrentCheckNesting = _lSaveNesting;	\
	  _lNested = false; \
	}

#else

  #define _mBeginLoopInvariant						_mNoact
  #define _mCheckLoopInvariant(condition, position)	_mNoact

#endif

//-----------------------------------------------
// Class '_xAssert' - assertion failure exception
//-----------------------------------------------

#if !defined(NDEBUG)

  class _xAssert : public _xDebug
  {
	protected:
	  void appendHeader(_eWorkspace ws, _eSize& size) const
	  {
		  append(ws, _mCstr("Assertion failed\n"), size);
	  }

	public:
	  _xAssert(const _eCallStackInfo& info, _eCstr file_name, _eNat line)
		  : _xDebug(info, file_name, line)
	  {
	  }
  };

  // Macros to perform a method post-assertion check
  #define _mBeginPostAssert \
	if (_tCall.enablePostAssert && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting)	\
		_mBeginCheck

  #define _mCheckPostAssert(condition, position) \
  	  if (!(condition)) throw _xAssert(_eCallStackInfo(_tSelf.functionname, _mCstr(position)), _mCurrentFile, __LINE__);	\
	  _mEndCheck

  // Macro used to disable assertion checking on a "super" call
  // NB - the method call immediately following use of this macro has its assertion checks disabled
  #define _mSuper		_eEscherRuntime::notSuper = false

  // Macros to perform an assertion check (not a method post-assertion)
  #define _mBeginEmbeddedAssert \
	if (_eEscherRuntime::EnableEmbeddedAssert && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting)	\
		_mBeginCheck

  #define _mCheckEmbeddedAssert(condition, position) \
  	  if (!(condition)) throw _xAssert(_eCallStackInfo(_tSelf.functionname, _mCstr(position)), _mCurrentFile, __LINE__);	\
	  _mEndCheck

#else

  #define _mBeginPostAssert								_mNoact
  #define _mCheckPostAssert(condition, position)		_mNoact
  #define _mSuper	 									(void)(0)
  #define _mBeginEmbeddedAssert							_mNoact
  #define _mCheckEmbeddedAssert(condition, position)	_mNoact

#endif

//---------------------------------------------------------
// Class '_xNoChoice' - choice-expression failure exception
//---------------------------------------------------------

class _xNoChoice : public _xDebug
{
  protected:
	void appendHeader(_eWorkspace ws, _eSize& size) const
	{
		append(ws, _mCstr("Choice expression failed to select option\n"), size);
	}

  public:
  #if _dCallStack
	_xNoChoice(const _eCallStackInfo& info, const _eNativeChar * file_name, const int line)
		: _xDebug(info, file_name, line)
	{
	}
  #else
	_xNoChoice(const _eNativeChar* file_name, const int line)
    	: _xDebug(file_name, line)
	{
	}
  #endif
};

//-----------------------------------------------------
// Macro '_mNoChoice' - Choice-expression-failure macro
//-----------------------------------------------------

#if _dCallStack

  #define _mNoChoice(position)	\
		throw _xNoChoice(_eCallStackInfo(_tSelf.functionname, _mCstr(position)), _mCurrentFile, __LINE__)

#else

  #define _mNoChoice(position)	\
		throw _xNoChoice(_mCurrentFile, __LINE__)

#endif

//-----------------------------------------------------------
// Class '_xLastChoice' - if-else-statement failure exception
//-----------------------------------------------------------

#if _dLastChoice

  class _xLastChoice : public _xDebug
  {
	protected:
	  void appendHeader(_eWorkspace ws, _eSize& size) const
	  {
		  append(ws, _mCstr("if-statement failed to select option\n"), size);
	  }

	public:
	  _xLastChoice(const _eCallStackInfo& info, _eCstr file_name, _eNat line)
	  	: _xDebug(info, file_name, line)
	  {
	  }
  };

  // Macro for last choice check ...
  #if defined(_MSC_VER) || defined(__BCPLUSPLUS__) || defined(__ghs__) || defined(__INTEL_COMPILER)
	// The following compilers can't handle a throw in a choice-expression:
	// Microsoft VC++ 6.0 SP5, Borland C++ 5.5.1, GreenHills C++, Intel 5.01

	// Helper function for last-choice check
	inline void _lThrowLastChoice(_xLastChoice e)
	{
		throw e;
	}

    #define _mCheckLastChoice(position, condition, expr) 	\
	  ( ((!_eEscherRuntime::EnableLastChoice || (condition)) ? (void)(0) : _lThrowLastChoice(_xLastChoice(_eCallStackInfo(_tSelf.functionname, _mCstr(position)) , _mCurrentFile, __LINE__))), \
		expr                                            	\
	  )

  #else

    #define _mCheckLastChoice(position, condition, expr) \
	  ((!_eEscherRuntime::EnableLastChoice || (condition)) ? expr : throw _xLastChoice(_eCallStackInfo(_tSelf.functionname, _mCstr(position)) , _mCurrentFile, __LINE__))

  #endif

#else

  #define _mCheckLastChoice(position, condition, expr)	(expr)

#endif

//--------------------------------------------------------------------
// Class '_xClassInvariant' - class-invariant failure class and macros
//--------------------------------------------------------------------

#if !defined(NDEBUG)

  class _xClassInvariant : public _xDebug
  {
	protected:
	  void appendHeader(_eWorkspace ws, _eSize& size) const
	  {
		  append(ws, _mCstr("Class Invariant failed\n"), size);
	  }

   #if _dCallStack

	  _eCallStackInfo Declaration;

	  void appendMessage(_eWorkspace ws, _eSize& size) const
	  {
		  append(ws, _mCstr("declared :"), size);
		  Declaration.appendSelf(ws, size);
		  append(ws, _mCstr("\nfailed :"), size);
		  CallStackInfo.appendSelf(ws, size);
	  }

	public:
	  _xClassInvariant(const _eCallStackInfo& decl, const _eCallStackInfo& info, _eCstr file_name, _eNat line)
		: _xDebug(info, file_name, line), Declaration(decl)
	  {
	  }

   #else

	public:
	  _xClassInvariant(_eCstr file_name, _eNat line)
		  : _xDebug(file_name, line)
	  {
	  }

   #endif
  };

  #if _dCallStack

    #define _mDeclareClassInvariant \
		protected: void _lClassInvariantCheck(const _eCallStackInfo& _vInfo, _eCstr _vFile, _eNat _vLine) const;

    // This macro is used to start the definition of invariant checking
    // It has logic in it to stop recursive calls.
    #define _mBeginClassInvariant(_vClassName) \
 	  void _vClassName::_lClassInvariantCheck(const _eCallStackInfo& _vInfo, _eCstr _vFile, _eNat _vLine) const \
	  { static _eMethod _tSelf(_mCstr("fun: class invariant check"), _mCurrentFile, __LINE__); \
	  	_eCallStack _tCall(&_tSelf);   \
	  	static _eBool _tNoRec = false; 		\
        if (_tNoRec) return; 				\
        _tNoRec = true;						\
		const _eNativeChar * _tClassName = _mCstr(#_vClassName);

	#define _mBeginClassInvariantItem \
	  try {

    #define _mCheckClassInvariantItem(_vCondition, position) \
		if (!(_vCondition)) throw _xClassInvariant(_eCallStackInfo(_tClassName, _mCstr(position)), _vInfo, _vFile, _vLine); \
	  } catch (_xCannotEvaluate) {}

  #else

    #define _mDeclareClassInvariant \
		protected: void _lClassInvariantCheck(_eCstr _vFile, _eNat _vLine) const;

    // This macro is used to start the definition of invariant checking
    // It has logic in it to stop recursive calls.
    #define _mBeginClassInvariant(_vClassName) \
 	  void _vClassName::_lClassInvariantCheck(_eCstr _vFile, _eNat _vLine) const \
	  { static _eBool _tNoRec = false; 	\
        if (_tNoRec) return; 	 		\
        _tNoRec = true;

	#define _mBeginClassInvariantItem \
	  try {

    #define _mCheckClassInvariantItem(_vCondition, position) \
		if (!(_vCondition)) throw _xClassInvariant(_vInfo, _vFile, _vLine); \
		} catch (_xCannotEvaluate) {}

  #endif

  #define _mInheritClassInvariant(ancestor) \
        ancestor::_lClassInvariantCheck(_vInfo, _vFile, _vLine);

  #define _mEndClassInvariant \
      _tNoRec = false; \
	}

  // Macro to check a class invariant
  #define _mCheckClassInvariant(position) \
	if (_eEscherRuntime::EnableClassInvariant && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting)	\
	{ ++_eEscherRuntime::CurrentCheckNesting;	\
	  _lClassInvariantCheck(_eCallStackInfo(_tSelf.functionname, _mCstr(position)), _mCurrentFile, __LINE__);	\
	  --_eEscherRuntime::CurrentCheckNesting;	\
	}

#else

  // Even if class invariant checking is disabled, we still have to generate something because
  // the "_mBeginClassInvariant" may be preceded by "template<class X>" or similar
  #define _mDeclareClassInvariant \
	protected: void _lClassInvariantCheck(_eCstr _vFile, _eNat _vLine) const;

  #define _mBeginClassInvariant(_vClassName) \
 	void _vClassName::_lClassInvariantCheck(_eCstr _vFile, _eNat _vLine) const \
	{

  #define _mEndClassInvariant \
    }

  #define _mInheritClassInvariant(ancestor)					_mNoact
  #define _mCheckClassInvariantItem(_vCondition, position)	_mNoact
  #define _mCheckClassInvariant(position)					_mNoact

#endif

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
