// **************************************************************************
// * File:   Environment.hpp												*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Code to implement interface to the local OS environment (ie.	*
// *         the local filesystem)											*
// **************************************************************************

#if !defined(_h_Main_Environment)
#define _h_Main_Environment

//---------
// Includes
//---------

#include <cstdio>						// for 'FILE' structure
#include "EnvironmentSupport_1.hpp"		// for various support classes ('FilePath' etc)

#if _dProfile
 #include "CpuTimer.hpp"
#endif

//--------------------------------
// Forward declarations of classes
//--------------------------------

//---------------------------------------------------------
// Class 'FileRef' - wrapper for the O/S level file pointer
//---------------------------------------------------------

class FileRef : public _eAny
{
  public:
	// extra-class-data macro (mClass_CI(FileRef))
    _mDefineType(FileRef)
    _mDefineCopy(FileRef)
    _mEqualsOperator(FileRef)
    _mPerfectTypeInfoHdrNA(FileRef)
    _mNew(FileRef)

  private:
	FILE*  f;
	_eBool writable;

  public:
	mutable _eBool lastOpWasWrite;

	//-------------
	// Constructors
	//-------------

	FileRef(FILE *ff, _eBool ww) : f(ff), writable(ww), lastOpWasWrite(ww)
	{
		_mBeginPre _mCheckPre(f != NULL, "0,0")
	}

	//----------------------
	// General functionality
	//----------------------

	_eBool fileWritable() const
	{
		return writable;
	}

	FILE* getFileHandle() const
	{
		return f;
	}

	_eBool operator==(const FileRef * a) const
	{
		return f == a->f;
	}
};

//--------------------------------------------------------------------
// Class 'Environment' - see Environment.cpp for function descriptions
//--------------------------------------------------------------------

class Environment : public _eAny
{
  private:
	void genEightBitSeq(const _eInt i, const _eInt numBytes, _eByte* buffer) const;

  public:
	// extra-class-data macro (mClass_CI(Environment))
    _mDefineType(Environment)
    _mDefineCopy(Environment)
    _mEqualsOperator(Environment)
    _mPerfectTypeInfoHdrNA(Environment)
    _mNew(Environment)

	// Constructors
	explicit Environment(const _eNativeChar * pathname);

	// Equality operator. Since a program can never have two different environments we
	// can just return true, rather than checking all paramenters in the environment.
   	_eBool operator==(const Environment * a) const
	{
		return true;
	}

	// File system functionality
	_eHndl<_eStandardInputStream> stdIn() const;
	_eHndl<_eStandardOutputStream> stdOut() const;
	_eHndl<_eStandardOutputStream> stdErr() const;
	void open(const _rstring filename, const _eSet < FileModeType :: _eEnum > mode, _eUnion &result);
    void open(const _eSeq < _eChar > hostName, const _eInt port, const SocketMode::_eEnum sktMode, _eUnion &result);
    void open(const _eSeq < _eByte > hostAddress, const _eInt port, const SocketMode::_eEnum sktMode, _eUnion &result);
	void makeDirectory(const _rstring pathname, FileError :: _eEnum & err);
	void readLine(_eSeq < _eChar > & line, FileError :: _eEnum & ret);
    void readLine(const FileRef* f, _eSeq<_eChar>& line, FileError :: _eEnum& ret);
	void move(const _eSeq<_eChar> oldPath, const _eSeq<_eChar> newPath, const _eBool clobber, FileError :: _eEnum &res);
	void flush(const FileRef* f, FileError :: _eEnum &ret);
	void close(const FileRef* f, FileError :: _eEnum &ret);
	void _rdelete(const _rstring pathname, FileError :: _eEnum &ret);	// note that in builtin.pd this is called "delete"

	void seek(const FileRef* f, _eNat filsSize, FileError :: _eEnum &ret);
    void fastForward (const FileRef * f, FileError :: _eEnum &ret);
    void rewind (const FileRef * f, FileError :: _eEnum &ret);

	_eBool fileValid(const _rstring pathname) const;
	void fileSize(const FileRef* f, _eNat &size, FileError :: _eEnum &ret) const;
	_eUnion tell(const FileRef* f) const;
	_eUnion fileStatus(const _rstring pathname) const;
	void setMode(const _rstring pathname, const _eSet<FileAttribute :: _eEnum> atts, FileError :: _eEnum &ret);
	void forceFileTime (const _eSeq < _eChar > filename, const Time * modified, const Time * accessed, FileError :: _eEnum & res);

	// Schemas to write 'Perfect' character types to the console..
	void print(const _rstring s);
	void printStdErr(const _rstring s);
	void print(const _eChar c);

	// Schemas to write 'Perfect' character types to a file..
	void print(const FileRef* f, const _rstring s, FileError :: _eEnum &ret);
	void print(const FileRef* f, const _eChar c, FileError :: _eEnum &ret);

	// Schemas to read 'Perfect' character types from a file..
	void scan(const FileRef* f, _rstring &s, const _eInt len, FileError :: _eEnum &ret);
	void scan(const FileRef* f, _eChar &c, FileError :: _eEnum &ret);

	// Schemas to write 'Perfect' types in binary format to a file
	void write(const FileRef* f, const _eSeq<_eByte> s, FileError :: _eEnum &ret);
	void write(const FileRef* f, const _eInt n, const _eInt numBytes, FileError :: _eEnum &ret);
	void write(const FileRef* f, const _eInt n, FileError :: _eEnum &ret);
	void write(const FileRef* f, const _eReal r, FileError :: _eEnum &ret);
	void write(const FileRef* f, const _eByte b, FileError :: _eEnum &ret);

	// Schemas to read 'Perfect' types in binary format from a file
	void read(const FileRef* f, _eSeq<_eByte> &s, const _eInt numBytes, FileError :: _eEnum &ret);
	void read(const FileRef* f, _eInt &n, const _eInt numBytes, FileError :: _eEnum &ret);
	void read(const FileRef* f, _eInt &n, FileError :: _eEnum &ret);
	void read(const FileRef* f, _eReal &r, FileError :: _eEnum &ret);
	void read(const FileRef* f, _eByte &b, FileError :: _eEnum &ret);

	// Other functionality ...
	_rstring getImagePath() const;
	_eUnion getCurrentDirectory() const;	// returns an _rstring || void
	void setCurrentDirectory(const _rstring path, FileError :: _eEnum & err);
	_eHndlUnionVoid<FilePath> normalizeFile(const _rstring basepath, const _rstring secondarypath) const;
	static _eBool caseSensitiveFileNames();
	static _eChar pathSeparator();
	void setCurrentThreadPriority(const _eNat priority);	// 1 to 20, where 1 is lowest priority
	_eNat clock() const;
	_eNat clocksPerSecond() const;
	_eHndl < Time > getCurrentDateTime() const;	// return date and time in binary format - see StringUtils class for formatting functions
	_eUnion getEnvironmentVar(const _rstring varName) const;
	_eSeq < _eInt > getImageVersion(const _rstring moduleName = _rstring()) const;
	_eNat getMemoryUsed() const;			// return total number of bytes allocated.
	_rstring getRuntimeLanguage() const;
	void garbageCollect() const				// instruct the runtime system to free up unused memory
		{ _eMem::garbageCollect(); }		// forward the call on to the memory management system
	_eHndl<OsInfo> getOsInfo() const;
    void execute(const _rstring command, const _eSeq < _rstring > args, const FileRef* stdinfile,
					const FileRef* stdoutfile, const FileRef* stderrfile, FileError :: _eEnum & err);

	// Runtime option functionality
    void setRuntimeOptionState (const DebugType :: _eEnum, const _eBool state);
    void setRuntimeOptionsState (const _eSet < DebugType :: _eEnum >, const _eBool state);
    void setRuntimeOption (const DebugType :: _eEnum);
    void setRuntimeOptions (const _eSet < DebugType :: _eEnum >);
    void clrRuntimeOption (const DebugType :: _eEnum);
    void clrRuntimeOptions (const _eSet < DebugType :: _eEnum >);
	void setMaxCheckNestLevel (const _eNat);

	void startProfiling();
	void profile(const FileRef* f, FileError :: _eEnum &ret);

	//--------------------------------------------------------------
	// Private data
	//--------------------------------------------------------------

  private:
	mutable _eHndl<_eStandardInputStream> stdinHandle;
	mutable _eHndl<_eStandardOutputStream> stdoutHandle, stderrHandle;
	_rstring ImagePath;						// the full-path to the compiler executable image
};

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
