// **************************************************************************
// * File:   CallStack.hpp										   			*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Definition of global call-stack for runtime de-bugging (call-	*
// *         stack is displayed after an exception).						*
// **************************************************************************

#if !defined(_h_callstack)
#define _h_callstack

#if _dProfile
 #include "CpuTimer.hpp"
 #include <cstdio>					// for class FILE
#endif


//-----------------------------------------------
// Forward declarations and external dependancies
//-----------------------------------------------

class _eCallStack;
class _eMethod;

namespace _eEscherRuntime
{
	extern _eCallStack * threadCallStack;
	extern _eMethod const *listOfMethods;
}

//---------------------------------------------------
// Class '_eMethod'
// This class holds static data relating to a method
//---------------------------------------------------

class _eMethod
{
  public:

#if _dProfile
	// Counters for profiling purposes
	// Put these before other data for optimum packing, because they are each 8 bytes long
	_eCpuTimer timeInMethod;		                            // time spent in the method and its children
	_eCpuTimer timeReallyInMethod;								// time spent in this method, ecluding calls from it
#endif

	// The following data is all public, but since it is all declared 'const', nothing can modify it
	_eMethod const * const next;								// link to other method descriptors
	_eNativeChar const * const functionname;							// name of this method
	_eNativeChar const * const filename;								// filename we are declared in
	int const linenumber;										// line number within the file

    //---------------------------
    // Formatted output function
    //---------------------------
    _eNativeChar const *formattedStr() const;

#if _dProfile
    //---------------------------
	// Profile output function
    //---------------------------
	int writeProfileInfo(FILE *f, _eCpuTimer quantum) const;
#endif

	// Constructor
	_eMethod(_eNativeChar const *name, _eNativeChar const *file, int lnumber)
		:
#if _dProfile
				timeInMethod(0), timeReallyInMethod(0),
#endif
				next(_eEscherRuntime::listOfMethods),
				functionname(name), filename(file), linenumber(lnumber)
	{
		_eEscherRuntime::listOfMethods = this;					// link this method in to the list
	}
};

//---------------------------
// Code - class '_eCallStack'
//---------------------------

class _eCallStack
{
#if _dProfile
  private:
	// put these first for optimum packing as they are 8 bytes long
	_eCpuTimer timeOfCall;										// the time at which this call started
	_eCpuTimer totalBeforeCall;									// the total before this call started
	_eCpuTimer timeInChildren;									// time spent in children

	static _eCpuTimer profileStartTime;							// the time at which we started profiling
	static _eCpuTimer totalProfilingTime;						// the total time for which profiling has been enabled
	static bool profilingEnabled;								// whether profiling is enabled

	void beginCall();
	void beginCall(_eCpuTimer);

	void endCall();
	void endCall(_eCpuTimer);

  public:
	// Method to reset and start the profile counters
	static void startProfiling();

	// Method to stop and update the profile counters ready to output profile info
	static void stopProfiling();

	// Method to get the total time for which profiling has been enabled
	static _eCpuTimer getTotalProfilingTime()
	{
		return totalProfilingTime;
	}

#endif

  private:
	_eMethod * const method;

  public:
	_eBool const enablePostAssert;
	_eCallStack * const next;

  public:
	_eNativeChar const * formattedStr() const
	{
		return method->formattedStr();
	}

	//---------------------------
	// Constructor
	//---------------------------
	_eCallStack(_eMethod * m)
			: method(m),  										// save location info
				enablePostAssert(_eEscherRuntime::EnablePostAssert && _eEscherRuntime::notSuper),
																// capture 'enable assertion checks' state
				next(_eEscherRuntime::threadCallStack)			// point to object at top of call stack
	{
#if _dProfile
		beginCall();											// this initialises the 3 timer fields
#endif
		_eEscherRuntime::threadCallStack = this;				// put us at top of call stack
		_eEscherRuntime::notSuper = true;						// restore normal state of flag
	}

	//---------------------------
	// Destructor
	//---------------------------
	~_eCallStack()
	{
#if _dProfile
		if (profilingEnabled)
		{
			endCall();
		}
#endif
		_eEscherRuntime::threadCallStack = next;				// remove this object from head of call stack
	}
};

#if _dCallStack

  // Macros to provide function/operator/constructor naming and
  // call-stack information at run-time...
  #define _mFunction( name ) static _eMethod _tSelf(_mCstr("fun: ") _mCstr(#name), _mCurrentFile, __LINE__); _eCallStack _tCall(&_tSelf)
  #define _mSchema( name )   static _eMethod _tSelf(_mCstr("sch: ") _mCstr(#name), _mCurrentFile, __LINE__); _eCallStack _tCall(&_tSelf)
  #define _mOperator( name ) static _eMethod _tSelf(_mCstr("opr: ") _mCstr(#name), _mCurrentFile, __LINE__); _eCallStack _tCall(&_tSelf)
  #define _mSelector( name ) static _eMethod _tSelf(_mCstr("sel: ") _mCstr(#name), _mCurrentFile, __LINE__); _eCallStack _tCall(&_tSelf)
  #define _mBuild			 static _eMethod _tSelf(_mCstr("bld"), 		   		   _mCurrentFile, __LINE__); _eCallStack _tCall(&_tSelf)
  // Global _tSelf for use in constructor inherits and member initialisation code
  extern _eMethod _tSelf;

#else

  #define _mFunction( name ) _mNoact
  #define _mSelector( name ) _mNoact
  #define _mSchema( name )	 _mNoact
  #define _mOperator( name ) _mNoact
  #define _mBuild			 _mNoact

#endif

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
