// **************************************************************************
// * File:   AnyBase.hpp													*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Definition of super class for all handle-aware classes		 	*
// *         (_eAnybase)													*
// **************************************************************************

#if !defined(_h_AnyBase)
#define _h_AnyBase 1

#include <cassert>

#include "Types.hpp"
#include "Order.hpp"

class _eFullTypeInfo;	// we forward declare this so that the '_aGetFullTypeInfo()' function can return a pointer to it.

class _eAnyBase
{
  private :
	mutable _eUseCount _lCounter;
  public :

	//----------------------------
	// Constructors and destructor
	//----------------------------

	// Default constructor (note: share count is zero)
	_eAnyBase() : _lCounter(0)
	{
	}

	// Copy constructor (note: sharing count is set to zero)
	_eAnyBase(const _eAnyBase&): _lCounter(0)
	{
	}

	// Destructor - must be virtual since every generated class inherits
	// from this class...
	virtual ~_eAnyBase()
	{
	}

	//------------------------------
	// Concrete member functionality
	//------------------------------

	// Increment the sharing count
	// NB - this must be atomic if multi-threading is supported!
	void _lIncrementCounter() const
	// pre _lCounter >0 0
	{
		assert(_lCounter >= 0);
		++_lCounter;
	}

	// Set the use count to 1. Used on objects we have just created (more efficient than incrementing the use count).
	void _lInitCounter() const
	// pre _lCounter == 0
	{
		assert(_lCounter == 0);
		_lCounter = 1;
	}

	// Decrement the sharing count, returning "true" if sharing count reaches 0
	// NB - this must be atomic if multi-threading is supported!
	_eBool _lDecrementCounter() const
	// pre _lCounter > 0
	{
		assert(_lCounter > 0);
		return (--_lCounter) == 0;
	}

	// Test whether this object is shared (used to decide whether to unbind)
	_eBool _lIsShared() const
	{
		return _lCounter > 1;
	}

	// Test whether this object is sharable
	// The 'self' object in a schema that modifies self has already been unbound so it not sharable
	_eBool _lNotSharable() const
	{
		return _lCounter < 0;
	}

	// Make a sharable object unsharable
	// Used to make '*this' unsharable on entry to a schema that modifies 'self'
	// Note: Exceptionally, the use count may be greater than one on entry if a temporary object has been created
	// when evaluating an argument, e.g. in "x[n]!s(x[n].a)" then the use count of 'self' may be 2 on entry to s,
	// depending on when the temporary handle to x[n] used to evaluate x[n].a gets destroyed.
	// For this reason, we return the original value so that it can be restored.
	void _lDontShare()
	{
		_lCounter += _lUnsharableValue;
	}

	// Make an unsharable object sharable. We assume the use count was 1 when we made it unsharable.
	// Used to make '*this' sharable on exit from a schema that modifies 'self'
	void _lReShare()
	// pre _lCounter < 0
	{
		assert(_lCounter < 0);
		_lCounter -= _lUnsharableValue;
	}

	// Get relative rank between 2 classes
	_eRank :: _eEnum _lClassRank(const _eAnyBase*) const;

	// Default relative rank operator - overridden in any class that defines "operator ~~"
	virtual _eRank :: _eEnum _oRank(const _eAnyBase* arg) const
	{
		return _lClassRank(arg);
	}

	// Less than operator is defined in terms of rank
	_eBool _oLess (const _eAnyBase* arg) const
	{
		return _oRank(arg) == _eRank :: below;
	}

	// This function is overridden in every class and provides a way to recurse up the inheritance tree. This verion
	// returns null to indicate that we have reached the top of the tree. Note that it MUST be virtual because the code
	// in _lClassRank calls it on the _eAnyBase pointers who's rank it's evaluating
	virtual const _eFullTypeInfo* _aGetFullTypeInfo() const
	{
#if defined(__BCPLUSPLUS__)
		// Borland C++ 5.5 refuses to allow "static_cast<X>(0)" if X is not completely defined, so use this instead...
		return (const _eFullTypeInfo*) 0;
#else
		return _mNullPtr(_eFullTypeInfo);
#endif
	}

	//------------------------------
	// Deferred member functionality
	//------------------------------

	// Create a new copy of the entire object
	virtual _eAnyBase* _lCopy() const = 0;

	// Return the type of the object
	virtual const std::type_info& _lType() const = 0;

	// Check whether this object is equal to another (ignoring the sharing count)
	virtual _eBool _lEqual(const _eAnyBase* h) const = 0;

	// Dummy schemas to allow us to call _astore(..)/_afill(..) on handles without the problems associated with
	// re-interpret cast (see the comment in BaseHandle.e)
	virtual void _astore(_eStoreStream* stream) const {}
	virtual void _afill(_eFillStream* stream) {}
};

//-----------------------------------------------------
// Global equality operator - called by member equality
// operators for handle classes that might be null
//-----------------------------------------------------

inline _eBool _lEqual(const _eAnyBase* arg1, const _eAnyBase* arg2)
{
	return     (arg1 == NULL && arg2 == NULL)
			|| (arg1 != NULL && arg1->_lEqual(arg2));
}

//---------------------------------------------------------------------------------
// Global rank operator - called when the first operand might be null
//---------------------------------------------------------------------------------

inline _eRank :: _eEnum _onRank(const _eAnyBase* arg1, const _eAnyBase* arg2)
{
	return (arg1 != NULL) ? arg1->_oRank(arg2)
			: (arg2 != NULL) ? _eRank :: below
				: _eRank :: same;
}

//---------------------------------------------------------------------------------
// Global less-than operator - called when the first operand might be null
//---------------------------------------------------------------------------------

// Less than operator is defined in terms of rank
inline _eBool _onLess (const _eAnyBase* arg1, const _eAnyBase* arg2)
{
	return _onRank(arg1, arg2) == _eRank :: below;
}

//---------------------------------------------------------------------------------
// Function used to compare two objects (used in set, bag, sequence and mapping to
// optimise equality operator). Returns true if both operands have the same address
//---------------------------------------------------------------------------------

inline _eBool _lSameObject(const _eAnyBase* arg1, const _eAnyBase* arg2)
{
	return arg1 == arg2;
}

//----------------------------------------------------------------------------
// Class used in schemas that modify 'self' to make the self-object unsharable
//----------------------------------------------------------------------------

class _eDontShare
{
private:
	_eAnyBase *ptr;												// pointer to object we don't wish to share

public:
	// Constructor sets is target to "not sharable"
	_eDontShare(_eAnyBase* p)
	{
		if (p->_lNotSharable())					                // if object is already unsharable
		{
			ptr = NULL;	                                        // don't do anything
		}
		else
		{
			p->_lDontShare();               					// mark object unsharable
			ptr = p;                                            // remember to mark it sharable later
		}
	}

	// Destructor resets its target to "sharable"
	~_eDontShare()
	{
		if (ptr != NULL)										// if we marked the object unsharable
		{
			assert(ptr->_lNotSharable());
			ptr->_lReShare();									// mark object sharable
		}
	}
};

// Macro used at start of modifying schemas to prevent sharing of an object (usually 'self')
// The parameter must be 'this' or a simple variable parameter name
// A semicolon should be generated after the macro
#define _mDontShare(x)		_eDontShare _lDontShare__ ## x(x)

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
