// **************************************************************************
// * File:   String.cpp										   				*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Code to implement string representation, and conversion		*
// *         routines between C and Perfect representations.				*
// * Notes:  Type '_rstring' is a Perfect representation of a string, ie.	*
// *         it is a sequence of char.										*
// *         Global routines 'toString', '_lString' and 'toNat' convert an	*
// *         integer to a string; Convert a C string into a Perfect			*
// *         string; and convert a Perfect string into a natural			*
// *         respectively.													*
// **************************************************************************

//---------
// Includes
//---------

#include "Ertsys.hpp"
#include "CharHelpers.hpp"

#include <cstring>			// for strlen
#if defined(_dUnicode)
  #include <cwchar>			// for wcslen etc.
  #if defined(__GNUC__)
    #include "LinuxSystemCalls.hpp"
  #endif
#endif

#include <cstdio>			// for sprintf
#include <cstdlib>			// for atoi


//--------------------------------
// Code for _eCstring Constructors
//--------------------------------

// Constructor to build a C string from a 'Perfect' string
_eCstring :: _eCstring(const _rstring& arg)
  : length(arg._oHash() + 1)
{
	s = static_cast<_eNativeChar *>(_eMem::alloc(length * sizeof(_eChar)));
	for (int i = 0; i < static_cast<int>(length) - 1; ++i)
	{
		s[i] = arg[i];
	}
	s[length - 1] = '\0';
}

// Constructor to build a C string from a const char*
_eCstring :: _eCstring(const _eNativeChar *arg)
  : length(::_mStrlen(arg) + 1)
{
	s = static_cast<_eNativeChar *>(_eMem::alloc(length * sizeof(_eChar)));
	memcpy(s, arg, length * sizeof(_eChar));
}

// Copy constructor
_eCstring :: _eCstring(const _eCstring& arg)
  : length(arg.length)
{
	s = static_cast<_eNativeChar *>(_eMem::alloc(length * sizeof(_eChar)));
	memcpy(s, arg.s, length * sizeof(_eChar));
}

// Constructor to build a string with an empty buffer of a specified size
_eCstring :: _eCstring(_eSize arg)
  : length(arg)
{
	s = static_cast<_eNativeChar *>(_eMem::alloc(length * sizeof(_eChar)));
}

//------------------------------------
// Code for other _eCstring operations
//------------------------------------

// Destroy the object, releasing the memory
_eCstring :: ~_eCstring()
{
	_eMem::free(s, length * sizeof(_eChar));
}

// Return the C string represented by this instance as a 'Perfect' string
_rstring _eCstring :: eselString() const
{
	return _lString(s);
}

// Return the strlen of the string
size_t _eCstring :: strlen() const
{
	return :: _mStrlen(s);
}

//---------------------------
// Global conversion routines
//---------------------------

// Int, real and byte 'toString' methods - these are not ideal as they have a 100 character limit,
// however, given the current implementations of int and real this is not actually a limitation at all.
#define bufferSize 100
_rstring _ltoString(const _eInt i)
{
	_eNativeChar buffer[bufferSize];
#if defined(_dUnicode)
	swprintf(buffer, bufferSize, L"%d", i);
#else
	sprintf(buffer, "%d", i);
#endif
	return _lString(buffer);
}
_rstring _ltoString(const _eReal r)
{
	_eNativeChar buffer[bufferSize];
#if defined(_dUnicode)
	swprintf(buffer, bufferSize, L"%g", r);
#else
	sprintf(buffer, "%g", r);
#endif
	// Unfortunately %g will print as an integer if the real happens to be an integer value.
	// We will add back a '.0' in this case to make it clear that this is a real.
	for (int i = 0; buffer[i]; i++)
	{	if (buffer[i] == '.' || buffer[i] == 'e' || buffer[i] == 'E')
			// Was already in 'real' form, so we can just output the result
			return _lString(buffer);
	}
	return _lString(buffer)._oPlusPlus(_mString(".0"));
}
_rstring _ltoString(const _eByte b)
{
	_eNativeChar buffer[bufferSize];
#if defined(_dUnicode)
	swprintf(buffer, bufferSize, L"%u", b);
#else
	sprintf(buffer, "%u", b);
#endif
	return _lString(buffer);
}
#undef bufferSize

// Convert an 'Perfect' char into a 'Perfect' string
_rstring _ltoString(const _eChar c)
{
	return _eSeq<_eChar>(c);
}

// Convert a 'Perfect' bool into a 'Perfect' string
_rstring _ltoString(const _eBool b)
{
	return b ? _mString("true") : _mString("false");
}

// Convert a null-terminated C string into a 'Perfect' string
_rstring _lString(const _eNativeChar *arg)
{
	return _eSeq<_eChar>(_mStrlen(arg), arg);
}

// Convert a 'Perfect' string into a 'Perfect' natural
_eNat _lstringToNat(const _rstring s)
{
    _mFunction (_lstringToNat);
#if !defined(NDEBUG)
    _mBeginPre _mCheckPre ((0 < s._oHash ()), "0,0");
    _mBeginPre
	_eBool allDigits = true;
	for (int i = 0; i < s._oHash() && allDigits; i++) 
	{
		allDigits = :: _lisDigit(s[i]);
	}
    _mCheckPre (allDigits, "0,0");
#endif

	const _eCstring temp(s);
	const _eNativeChar *str = temp.str();
	_eNat val = 0;
	while (*str >= _mCstr('0') && *str <= _mCstr('9'))
	{
		val = (val * 10) + (*str - _mCstr('0'));
		++str;
	}
	return val;
}

// End.
