// **************************************************************************
// * File:   Ertsys.cpp											   			*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Code to implement the main entry point into a Perfect program	*
// **************************************************************************

//---------
// Includes
//---------

#include "Ertsys.hpp"
#include "CharHelpers.hpp"

#if defined(_MSC_VER) && defined(_DEBUG)
  #include <crtdbg.h>
#endif

#include <new>			// for 'bad_alloc' exception class

#if defined(_MSC_VER)
#include <new.h>		// for '_set_new_handler' function
#endif

#include <cstdio> 		// for printf
#include <cstring>		// for strcpy
#include <cwchar>		// for wcscpy under Linux
#include <cstdlib>		// for mbstowcs

// Main entry point into a compiled 'Perfect' program
extern void _rmain (Environment*, const _eSeq < _rstring >, _eInt &);

//------------------------
// define global variables
//------------------------

namespace _eEscherRuntime
{
	//---------------------------------------------------------------------------
	// Runtime check flags.
	// Note that each of these will only be effective when running a debug
	// build that had the relevant check generated by the Perfect Developer Tool.
	//---------------------------------------------------------------------------

	_eBool EnablePre = true;
	_eBool EnablePost = true;
	_eBool EnableLoopInvariant = true;
	_eBool EnableLoopDecrease = true;
	_eBool EnableSpecDecrease = true;
	_eBool EnableImpDecrease = true;
	_eBool EnablePostAssert = true;
	_eBool EnableEmbeddedAssert = true;
	_eBool EnableLastChoice = true;
	_eBool EnableClassInvariant = true;
	_eBool EnableConstraints = true;

	// This flag is normally true but we set it to false just before making a 'super' call
	_eBool notSuper = true;

	// This count is used to allow us to abort nested checks for everything except loop-invariant
	// checks (these are NEVER allowed to do nested checks, else we would blow the stack!). An
	// example is where we have an expression that we test in an assertion; this may call a function
	// that itself has an assertion - this is the kind of nesting we are talking about.
	unsigned int MaxCheckNesting = 1;
	unsigned int CurrentCheckNesting = 0;

	// This flag allows us to not destroy the objects on any global heaps on program termination -
	// the operating system can take care of this for us.
	_eBool Terminated = false;

	const _eSize defaultMemoryAllocation = 0x100;
	const _eNativeChar *stringArray[maxStackOutput];

	const int EselExceptionFailureCode = 255;
	const int UxpdExceptionFailureCode = 254;
	const int AllocFailureCode = 253;

	#if defined(_MSC_VER) && _MSC_VER < 1400
 		int __cdecl failedAllocHandler(size_t size)
 		{
	 		throw std :: bad_alloc();
			return 0;		// to keep VC++ 6.0 happy
 		}
	#endif
}	                                                            // end namespace _eEscherRuntime

//-----
// Code
//-----

// Microsoft compiler specific note regarding fastcall and use of __cdecl:
// Although we use the /Gr option to specify the 'fastcall' calling convention, we don't need to use the __cdecl modifier
// for the 'main(..)' function, as stated in the MSDN help extract below (search for '__fastcall' in MSDN). In fact, we can't
// use the modifier, or the file won't compile in GCC. However, the Microsoft compiler gives us a warning so we have to test if
// we are the being compiled with Microsoft and define main's function header appropriately - sorry. MSDN extract follows :-
// "Using the /Gr compiler option causes each function in the module to compile as fastcall unless the function is declared
//  with a conflicting attribute, or the name of the function is main. "
#if defined(_MSC_VER)
int __cdecl main(int /*argc*/, char* argv[])
#else
int main(int /*argc*/, char* argv[])
#endif
{
	const _eNativeChar * memAllocErrorMessage = _mCstr("Memory allocation failure detected!\n");
	const _eSize bufferSize = 512;		// size of exception message buffer - must be big enough to hold the preceding message

	_eInt result;
	int hadException = 0;
	_eNativeChar buffer[bufferSize];

	//-----------
	// Debug code
	//-----------

  #if defined(_MSC_VER) && defined(_DEBUG)
    int tmpflag = _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG);
    tmpflag |= _CRTDBG_LEAK_CHECK_DF;
    _CrtSetDbgFlag(tmpflag);
  #endif

	//------------------------------------------------------------
	// Set-up the default exception handler for failed 'new' calls
	// We only do this under Microsoft, because failed new calls
	// return NULL by default. Under ANSI, a failed new should
	// cause a 'bad_alloc' exception to be thrown. We thus make
	// this the default behaviour under Microsoft also ...
	//------------------------------------------------------------

  #if defined(_MSC_VER) && _MSC_VER < 1400
	_set_new_handler(&_eEscherRuntime::failedAllocHandler);
  #endif

	//----------
	// Main code
	//----------

	try
	{
#if defined(_dUnicode)
		_eNativeChar* buf;
		ncs2wcs(buf, argv[0]);
		_eHndl <Environment> RuntimeEnvironment (new Environment(buf));
#else
		_eHndl <Environment> RuntimeEnvironment (new Environment(argv[0]));
#endif

		//----------------
		// parse arguments
		//----------------

		_eSeq<_eSeq<_eChar> > args;
		while (*argv != NULL)
		{
#if defined(_dUnicode)
			size_t len = strlen(*argv);
			_eNativeChar* buf = new _eNativeChar[len + 1];
			mbstowcs(buf, *argv, len + 1);
			args = args.append(_lString(buf));
			delete[] buf;
#else
			args = args.append(_lString(*argv));
#endif
			++argv;
		}

		//---------------------------------
		// Call the compiled 'Perfect' code
		//---------------------------------

		_rmain(RuntimeEnvironment.ChangePtr(), args, result);
	}

	//--------------------------------------------------------
	// Deal with any exception raised by execution of the code
	//--------------------------------------------------------

	catch (const std::bad_alloc&)
	{
		hadException = 1;
		_mStrcpy(buffer, memAllocErrorMessage);
		// Set our return value
		result = _eEscherRuntime::AllocFailureCode;
	}
	catch (const _xCeption& exception)
	{
		hadException = 1;
		// Get the message associated with the exception
		exception.getMessage(buffer, bufferSize);
		// Set our return value...
		result = _eEscherRuntime::EselExceptionFailureCode;
	}

	// If had an exception, print the message and dump any callstack we may have
	if (hadException)
	{
#if defined(_dUnicode)
		printf("%ls\n", buffer);
#else
		printf("%s\n", buffer);
#endif
		#if defined(_DEBUG)
			_eNat counter = 0;
			// Print out the call-stack...
			printf("\nCall-stack (most recent call first):\n\n");
			while(counter < _eEscherRuntime::maxStackOutput && _eEscherRuntime::stringArray[counter] != 0)
			{
#if defined(_dUnicode)
				printf("%ls\n", _eEscherRuntime::stringArray[counter]);
#else
				printf("%s\n", _eEscherRuntime::stringArray[counter]);
#endif
				counter++;
			}
		#endif
	}

	//----------------------------------
	// Return our result value to the OS
	//----------------------------------

	_eEscherRuntime :: Terminated = true;

	return static_cast<int>(result);
}

// End.
