// **************************************************************************
// * File:   ExceptionBase.hpp												*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Definition of the runtime exception base class					*
// *         (_Xception), and some of the low-level exception classes.		*
// *         Because these exceptions are used at the most fundamental		*
// *         levels we do not associate any dynamically stored data with it.*
// **************************************************************************

#if	!defined(_h_ExceptionBase)
#define _h_ExceptionBase

//-----------------------------------------------------------
// Class '_xCeption' - 'Perfect' runtime exception base class
//-----------------------------------------------------------

class _xCeption : public _eFormat
{
  public:
	//----------------------------
	// Constructors and destructor
	//----------------------------

	_xCeption(_eCstr, _eNat);

	virtual ~_xCeption(){}

	//------------------------------
	// Concrete member functionality
	//------------------------------

	// This sends the error message to the 'buffer' taken to
	// be of size 'size'.
	void getMessage(_eNativeChar * const buffer, const _eSize size) const;

  protected:

  #if _dCppInfo
	_eCppInfo cppInfo;
  #endif

	//------------------------------
	// Deferred member functionality
	//------------------------------

	// broad category of error
	virtual void appendHeader(_eWorkspace, _eSize&) const = 0;

	// detailed description
	virtual void appendMessage(_eWorkspace, _eSize&) const = 0;
};

//-----------------------------------------------------
// Class '_xCannotHappen' - for internal runtime errors
//-----------------------------------------------------

class _xCannotHappen : public _xCeption
{
  public:
	//------------
	// Constructor
	//------------

	_xCannotHappen(_eCstr description, _eCstr file, _eNat line)
	  : _xCeption(file, line), name(description){}

	//------------------------------
	// Concrete member functionality
	//------------------------------

	void appendHeader(_eWorkspace buffer, _eSize& size) const
	{
		append(buffer, _mCstr("runtime-system error\n"), size);
	}

	void appendMessage(_eWorkspace buffer, _eSize& size) const
	{
		append(buffer, name, size);
	}

  protected:
	_eCstr name;
};

//-----------------------------------------------------------------
// Macro '_mCannotHappen' to throw internal runtime error exception
//-----------------------------------------------------------------

#if _dCannotHappen
 #define _mCannotHappen(condition, description) \
	if (condition) throw _xCannotHappen(_mCstr(description), _mCurrentFile, __LINE__)
#else
 #define _mCannotHappen(condition, description) _mNoact
#endif

//----------------------------------------------
// Macro '_mCheckCast' to throw internal runtime
// error exception on cast failure
//----------------------------------------------

#if _dCheckCast
  #define _mCheckCast(condition, description) \
     ((condition) ? throw _xCannotHappen(_mCstr(description), _mCurrentFile, __LINE__) : (void)0)
#else
  #define _mCheckCast(exception, description) _mNoact
#endif

//---------------------------------------------------------
// Class '_xMemory' to deal with memory allocation checking
//---------------------------------------------------------

#if _dMemory

  class _xMemory : public _xCannotHappen
  {
    public:
	  _xMemory(_eCstr file, _eNat line);
  };

  #define _mCheckMemAlloc(pointer) \
	 if (pointer == NULL) throw	_xMemory(_mCurrentFile, __LINE__)

#else

  #define _mCheckMemAlloc(pointer) _mNoact

#endif

//-------------------------------------------------------------------
// Macro '_mUnreachable' to throw internal runtime error exception in
// the case of supposed un-reachable code getting executed. Note that
// this is never disabled...
//-------------------------------------------------------------------

#define _mUnreachable(description)		throw _xCannotHappen(_mCstr(description), _mCurrentFile, __LINE__)

//-------------------------------------------------------------------
// Macro '_mAbsurd' is used in the body of absurd methods
//-------------------------------------------------------------------

#define _mAbsurd	_mUnreachable("Absurd method called")

//-------------------------------------------------------
// Class _xCannotEvaluate and macro _mCannotEvaluate
// Used for ghost methods whose value cannot be evaluated
//-------------------------------------------------------

#if !defined(NDEBUG)

	class _xCannotEvaluate
	{
	    public:
	  		_xCannotEvaluate() {}
	};

	#define _mCannotEvaluate \
		throw _xCannotEvaluate()
	#define _mBeginCheck \
		{   ++_eEscherRuntime::CurrentCheckNesting; \
			try {
	#define _mEndCheck \
			} catch (_xCannotEvaluate) {} \
			--_eEscherRuntime::CurrentCheckNesting; \
		}

#else

  #define _mCannotEvaluate	_mUnreachable("Ghost method called")

#endif

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
